%%
% This script is to demonstrate the simulations in the paper: 
% J. Zhang, S. P. Chepuri, R. C. Hendriks, R. Heusdens, Microphone Subset
% Selection for MVDR Bemaformer Based Noise Reduction, IEEE/ACM Trans.
% Audio, Speech, Language Processing, 2018. (accepted)
% Contact: Jie Zhang (j.zhang-7@tudelft.nl), TU Delft, the Netherlands
%
% Requirements: RIR generator, e.g., https://www.audiolabs-erlangen.de/fau/professor/habets/software/rir-generator
% CVX toolbox: http://cvxr.com/cvx/download/, provided by S. Boyd at Stanford University
% Audio data, e.g., TIMIT database
% Here, we use the statistics in one frequency bin to show the performance,
% the output noise power or SNR should be averaged over all freqencies.

% Explanation for test_data.mat:
% M:     number of microphones
% L:     dimension of the environment
% FC:    location of the fusion center
% mpos:  positions of microphones
% sigpos:position of the target source
% npos1: position of the 1st interfering source
% npos2: position of the 2nd interfering source
% alpha: performance controllor, can be changed by user
% as:    acoustic transfer function
% c0:    squared distance from microphones to FC
% Rnn:   noise correlation matrix (M x M)
% Rxx:   correlation matrix of target signal (M x M)
% rgg:   wireless transmission range
% option:initialization shape in [line_segment, ray, point, two_point]

%% Part II: data-driven sensor selection to generate the plots in Figure 3

close all, clear all, clc,
load('test_data.mat')
rgg = 1; 
alpha = 0.9; 
option = 'point';
centre = [L(1)/2 L(2)/2 3];
init_point = FC;

mvdr = inv(Rnn)*as/(as'*inv(Rnn)*as); % classical MVDR
npower_mvdr = real(mvdr'*Rnn*mvdr); 
sel_set = [];
global_index = 0;
for k = 1:30 % greedy iteration
    new_set = initialize_set(mpos,sel_set,init_point,sigpos, rgg, option);
    K = length(new_set);
    p = zeros(M,1); p(new_set) =1;  Phi = diag(p);
    Phi( ~any(Phi,2), : ) = [];
    Rnnk = Phi*Rnn*Phi'; clear Phi p;

    b = min(abs(eig(Rnnk)))/2;
    S = Rnnk - b * eye(K);
    S = inv(S);
    iter_mvdr = inv(Rnnk)*as(new_set)/(as(new_set)'*inv(Rnnk)*as(new_set)); % MVDR using candidate set
    iter_npower_mvdr = real(iter_mvdr'*Rnnk*iter_mvdr);


    if global_index == 0
        lambda = alpha/iter_npower_mvdr;
        disp(['iter: k = ', num2str(k), ',  Local constraint !!!']);
    elseif global_index == 1
        lambda = alpha/npower_mvdr;
        disp(['iter: k = ', num2str(k), ',  Global constraint !!!']);
    end
    cvx_status = '';
    cvx_begin  SDP quiet
         variable p(K,1) 
         minimize ( sum(diag(p)*c0(new_set)) )
         subject to
             [S + 1/b*diag(p), S*as(new_set); as(new_set)'*S, as(new_set)'*S*as(new_set)-lambda] == hermitian_semidefinite(K+1)
              p <= 1
              p >= 0
     cvx_end

     se_set = random_rounding(p, Rnnk, as(new_set), 1/lambda);
     p = zeros(K,1); p(se_set)=1;
    sel_set = new_set(find(p==1)); 
    if global_index == 0
        global_index = surround(new_set, sel_set,mpos',FC',rgg);
    end

    figure(k) % to plot selected sensors for each iteration
    set(gca, 'fontsize', 14);
    grid on;
    hold on;
    xlabel('$x(m)$','FontSize',18,'Interpreter','Latex');
    ylabel('$y(m)$','FontSize',18,'Interpreter','Latex');
    title(['k = ', num2str(k)]);
    axis([-0.3 L(1)+0.3 -0.3 L(2)+0.3])
    plot(mpos(:,1),mpos(:,2), 'ok', 'MarkerSize',8);
    plot(sigpos(1), sigpos(2), 'or', 'MarkerSize',14,'MarkerFaceColor','r','markerEdgeColor','r');
    plot([npos1(1) npos2(1)], [npos1(2) npos2(2)], '*b', 'MarkerSize',14,'MarkerFaceColor','b','markerEdgeColor','b')
    plot(FC(1), FC(2), 'sk', 'MarkerSize',12,'MarkerFaceColor','k','markerEdgeColor','k')
    plot(mpos(new_set,1),mpos(new_set,2), 'pr', 'MarkerSize',6,'MarkerFaceColor','r','markerEdgeColor','r')
    plot(mpos(sel_set,1),mpos(sel_set,2), 'ob', 'MarkerSize',8,'MarkerFaceColor','b','markerEdgeColor','b')
    %  legend('sensors','source','interference','fusion center','active','selected','FontSize',14);
    box on;
    
end
