%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Article:      Submodular Detection for Correlated Measurements
% Section:      Uncommon Covariance
% Author:       Mario Coutino
% Affiliation:  TU Delft
% Date:         November 2016
% Conference:   N/A
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

clear all, close all

COV_TYPES = {'random', 'toeplitz', 'rtoeplitz', 'uniform', 'diagonal'};

% parameters for simulation
N = 100;
V = 1:N;

if N <= 15
    kVec = 1:N;
else
	kVec = 5:5:N;
end

V2 = V;
gB = [];

nIt = 1;
nMat = 3;

SUB_APPROX = 1;

KL_cvx = nan(N, nMat, nIt);
KL_supsub = nan(N, nMat, nIt);
KL_surrogate = nan(N, nMat, nIt);
KL_exha = nan(N, nMat, nIt);
KL_greedy = nan(N, nMat, nIt);

for nT = 3:nMat
    
    fprintf('\n==================================================\n')
    fprintf('\n Type of matrix %s\n',COV_TYPES{nT})
    fprintf('\n==================================================\n')
    
    for it = 1:nIt
        fprintf('\n Iteration: %d',it)
        fprintf('\n++++++++++++++++++++++++++++++++++++++++++++++++++\n')

        % covariance matrix
        E0 = genCovMat(N,COV_TYPES{nT});
        E1 = genCovMat(N,COV_TYPES{nT});
        
        [U1,L1] = eig(E1);
        E1s = U1*sqrt(L1);
        
        % covariance decomposition
%         [S0,a0,Sinv0,ainv0] = paramCov(E0);
        a0 = min(eig(E0))/3; ainv0 = a0^-1;
        S0 = E0 - a0*eye(N); Sinv0 = inv(S0);
        
%         [S1,a1,Sinv1,ainv1] = paramCov(E1);
        a1 = min(eig(E1))/3; ainv1 = a1^-1;
        S1 = E1 - a1*eye(N); Sinv1 = inv(S1);
        
        % helper for cvx syntaxis
        En = E1s'*Sinv0;
        Ein = inv(Sinv0*E1s);

        hSinv1 = sqrtm(Sinv1);
        hS0 = sqrtm(S0);
        
        % handlers
        [kF, kG] = sfo_fn_klDiv(E0,E1);
        kFG = sfo_fn_lincomb({kF, kG},[1,-1]);
        [kFs, kGs] = sfo_fn_klDiv(E0,E1,SUB_APPROX);
        
        I = eye(N);
        
        costFncPrev = 0;
        
        nCC = 100;
        tic
        
        for k = kVec
            fprintf('Number of Sensors...%d\n\n',k);
            fprintf('----------------------------\n\n')
            Dk = zeros(N);
            for cc = 1:nCC
                
                % concave convex procedure
                Ak = inv(I + ainv0*Dk*S0);
                f1_t1 = log(det(I + ainv0*hS0*Dk*hS0'));
                
                cvx_begin sdp quiet
                
                    variable w(N)
                    variable t
                    expression f0
                    expression f1
                    
                    % concave epigraph
                    f0 = -t + ...
                       log_det(I + ainv1*hSinv1*diag(w)*hSinv1');
                    % affine
                    f1 = f1_t1 + ...
                        trace(Ak*(ainv0*(diag(w)-Dk)*S0) );
                    
                    % cost
                    costFnc = f0 - f1;
                    maximize( costFnc )
                    subject to
                        ones(N,1)'*w == k;
                        w >= 0;
                        w <= 1;
                        
                        [Sinv0 + ainv0*diag(w) En';
                            En t*eye(N)] >= 0;
                
                cvx_end
            
                imprv = abs(costFnc - costFncPrev);
                fprintf('Convergence...%f\n\n',imprv);
                costFncPrev = costFnc;
                
                if imprv < 1e-3
                    break;
                end
                w;
                Dk = diag(w);
%                 keyboard()
            end
            cvx_Time(k,nT, it) = toc;
            [wsel] = randomized_rounding_toeplitz2(w,kFG,k);
            wsel_idx = find(wsel == 1);
                
            KL_cvx(k,nT, it) = kFG(wsel_idx);
        end
        
        % Greedy Heuristic
        tic
        A_kl = sfo_supsub(kG,kF,V,N);
        KL_supsub(:,nT,it) = kSetFunctionCost(kFG,A_kl');
        supsub_time(:,nT,it) = toc;
        
        tic
        A_kl_s = sfo_supsub(kGs,kFs,V,N);
        KL_surrogate(:,nT,it) = kSetFunctionCost(kFG,A_kl_s');
        surrogate_time(:,nT,it) = toc;
        
        if N <= 15
            for kk = 1:N
                A_kl_star = sfo_exhaustive_max(kFG,V,kk);
                KL_exha(kk,nT,it) = kFG(A_kl_star');
            end
        end
        
        tic
        A_kl_greedy = sfo_greedy_k(kFG,V,N);
        KL_greedy(:,nT,it) = kSetFunctionCost(kFG,A_kl_greedy');
        greedy_time(:,nT,it) = toc;
    
    end
end
%%
figure, 
p_cvx = plot(kVec,KL_cvx(kVec,nT),'ob','linewidth',1.1); hold on
p_supsub = plot(KL_supsub(:,nT),'--+g','linewidth',1.1);
p_surr = plot(KL_surrogate(:,nT),'m','linewidth',1.1);
p_gred = plot(KL_greedy(:,nT),'--r','linewidth',1.1);
set(gca,'fontsize',13)
if N <= 15
    p_exh = plot(KL_exha(:,nT),'k','linewidth',1.1);
    legend([p_cvx; p_supsub; p_surr; p_exh; p_gred],...
        {'CCP Pocedure ','SupSub Procedure', 'SubSub (Surrogate)','Exhaustive Search',...
        'KL-Greedy'},'location','nw')
else
    legend([p_cvx; p_supsub; p_surr; p_gred],...
        {'CCP Pocedure ','SupSub Procedure', 'SubSub (Surrogate)',...
        'KL-Greedy'},'location','nw')
end
xlabel('Number of Sensors [k]')
ylabel('KL-Divergence')
set(gca,'yscale','log')
grid on
% save('klResults', 'KL_cvx', 'KL_supsub', 'KL_surrogate', 'KL_exha', 'KL_greedy')
% save('klResultsSpaceLarge')
%%
%%
figure, 
p_cvx = plot(kVec,(cvx_Time(kVec,nT)),'ob','linewidth',1.1); hold on
p_supsub = plot(kVec,ones(size(kVec))*cumsum(supsub_time(:,nT)),'--+g','linewidth',1.1);
p_surr = plot(kVec,ones(size(kVec))*cumsum(surrogate_time(:,nT)),'om','linewidth',1.1);
p_gred = plot(kVec,ones(size(kVec))*cumsum(greedy_time(:,nT)),'--xr','linewidth',1.1);
set(gca,'fontsize',13)
if N <= 15
    p_exh = plot(KL_exha(:,nT),'k','linewidth',1.1);
    legend([p_cvx; p_supsub; p_surr; p_exh; p_gred],...
        {'CCP Pocedure ','SupSub Procedure', 'SubSub (Surrogate)','Exhaustive Search',...
        'KL-Greedy'},'location','nw')
else
    legend([p_cvx; p_supsub; p_surr; p_gred],...
        {'CCP Pocedure ','SupSub Procedure', 'SubSub (Surrogate)',...
        'KL-Greedy'},'location','nw')
end
xlabel('Number of Sensors [k]')
ylabel('Time [s]')
set(gca,'yscale','log')
grid on