%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Article:      Submodular Detection for Correlated Measurements
% Section:      Testing Time Greedy vs Convex (numVariables)
% Author:       Mario Coutino
% Affiliation:  TU Delft
% Data:         July 2017
% Conference:   TSP
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%fi%%%%%%%%%%%%%%%%
clear all;
close all;
% seed rndm
rng(0);

nIt = 100;
maxN = 100;
cvxTime = nan(maxN,nIt);
grdTime = nan(maxN,nIt);

warning off

for it = 1:nIt
    try
        if (mod(it,10) == 0)
            fprintf('\n Iteration: %d\n',it)
        end
        
        % CVX Method
        for N = 10:10:maxN
            
            % parameters for simulation
%             N = 50;
            V = 1:N;
            
            k = ceil(N/2);
            
            fprintf('\n N: %d',N)
            % mean vectors
            theta0 = randn(N,1);
            theta1 = randn(N,1);

            % difference vector
            m = theta1 - theta0;

            % covariance matrix
            E = gallery('toeppd',N);
            E = E/min(eig(E));                  % minimum eig = 1
        
            % true cost function
            f = @(A) getSNR(m(A),E(A,A));
            F = sfo_fn_wrapper(f);
        
            fw = @(A) -f(A);
            Fw = sfo_fn_wrapper(fw);
        
            % covariance decomposition
            tic
            a = 0.5; ainv = a^-1;
            S = E - a*eye(N);
            Sinv = inv(S);
        
            zVec = zeros(N,1);

            cvx_begin sdp quiet
                variable w(N)
                variable t
                minimize t
                subject to
                sum(w) ==  k;
                [Sinv + ainv*diag(w), m;
                    m' , t] >= semidefinite(N+1);
                w(:)<=1;
                w(:)>=0;
            cvx_end
            cvxTime(N,it) = toc;
            
            % greedy for the submodular cost function
            tic
            Ag = maxSubSNRGreedy(m,E,N);
            grdTime(N,it) = toc;
            
        end
        
    catch ME
        rethrow(ME)
        fprintf('\n\n')
    end
        
end
%%
save('testTimeSNRvsN')
close all;
mean_grdTime = nanmean(grdTime(10:10:maxN,:),2);
mean_cvxTime = nanmean(cvxTime(10:10:maxN,:),2);

p1 = semilogy(1:10:maxN,mean_grdTime,'-ob','linewidth',1.1); hold on;
p2 = semilogy(1:10:maxN,mean_cvxTime,'-xr','linewidth',1.1);
set(gca,'FontSize',13)
xlabel('Available Sensors [N]')
ylabel('Time [s]')  
legend([p1;p2], {'Convex Method; K= N/2','Greedy Method; K ={1,...,N}'})
% plotting