%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Article:      Submodular Detection for Correlated Measurements
% Section:      Uncommon Mean [MonteCarlo]
% Author:       Mario Coutino
% Affiliation:  TU Delft
% Data:         July 2017
% Conference:   TSP
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%fi%%%%%%%%%%%%%%%%

% seed rndm
rng(0);

% parameters for simulation
N = 15;
V = 1:N;

nIt = 1e3;

exSNR = nan(N, nIt);
tSNR = nan(N, nIt);
gSNR = nan(N, nIt);
cvxSNR = nan(N, nIt);
worstSNR = nan(N, nIt);

for it = 1:nIt
    try
        if (mod(it,10) == 0)
            fprintf('\n Iteration: %d\n',it)
        end
        % mean vectors
        theta0 = randn(N,1);
        theta1 = randn(N,1);

        % difference vector
        m = theta1 - theta0;

        % covariance matrix
        E = genCovMat(N,'rtoeplitz');

        % covariance decomposition
        [S,a,Sinv,ainv] = paramCov(E);
        
        % true cost function
        f = @(A) getSNR(m(A),E(A,A));
        F = sfo_fn_wrapper(f);
        
        fw = @(A) -f(A);
        Fw = sfo_fn_wrapper(fw);
        
        zVec = zeros(N,1);
                    
        for k = 1:N
            % exhaustive search
            Ae = sfo_exhaustive_max(F,V,k);
            exSNR(k,it) = F(Ae);
            % convex
            %cvx_solver sedumi
            cvx_begin sdp quiet
                variable w(N)
                variable t
                minimize t
                subject to
                sum(w) ==  k;
                [Sinv + ainv*diag(w), m;
                    m' , t] >= semidefinite(N+1);
                w(:)<=1;
                w(:)>=0;
            cvx_end
            
            [wsel] = randomized_rounding_toeplitz(w,S,k,theta1-theta0);
            wsel_idx = find(wsel == 1);
    
            cvxSNR(k,it) = m(wsel_idx)'*inv(E(wsel_idx,wsel_idx))*m(wsel_idx);
            
            %worst sensor selection
            Aworst = sfo_exhaustive_max(Fw,V,k);
            worstSNR(k,it) = F(Aworst);
        end
        
        % greedy for the true cost function
        At = sfo_greedy_k(F,V,N);
        tSNR(:,it) = kSetFunctionCost(F,At');
            
        % greedy for the submodular cost function
        Ag = maxSubSNRGreedy(m,E,N);
        gSNR(:,it) = kSetFunctionCost(F,Ag');
    catch
        disp('')
    end
        
end
% plotting
%%
mean_ExSNR = nanmean(exSNR,2);
mean_TrSNR = nanmean(tSNR,2);
mean_SuSNR = nanmean(gSNR,2);
mean_CXSNR = nanmean(cvxSNR,2);
mean_SNRworst = nanmean(worstSNR,2);

pi0 = 0.3;
pi1 = 1 - pi0;

gamma_e = log(pi0/pi1);

Pe_BA = @(sw) pi1.*(1 - qfunc(gamma_e./sqrt(sw) - sw./(2.*sqrt(sw)))) + ...
    pi0.*qfunc(gamma_e./sqrt(sw) + sw./(2*sqrt(sw)));

alpha = 0.01;
gamma = qfuncinv(alpha);

thr = @(sw) sw.*qfuncinv(alpha) - sqrt(sw)./2;
P_md = @(sw) 1 - qfunc( (2.*thr(sw) - sw)./(2.*sqrt(sw)) );

close all,
matType = 1;
bnd = (1-1/exp(1));

hfig = area([Pe_BA(mean_SNRworst(:,matType)) ...
    Pe_BA(mean_ExSNR(:,matType))-Pe_BA(mean_SNRworst(:,matType))],'LineStyle','none');
set(hfig(1),'FaceColor',[1 1 1])
set(hfig(2),'FaceColor',[0.9 0.9 0.9])
set(gca,'yscale','log','fontsize',13), hold on,

pl1 = semilogy(V,Pe_BA(mean_ExSNR(:,matType)),'-','linewidth',1.1);
pl2 = semilogy(V,Pe_BA(mean_TrSNR(:,matType)),'+k','linewidth',1.1);
pl3 = semilogy(V,Pe_BA(mean_SuSNR(:,matType)),'-dm','linewidth',1.1);
pl4 = semilogy(V,Pe_BA(mean_CXSNR(:,matType)),'-og','linewidth',1.1);
pl5 = semilogy(V,Pe_BA(mean_SNRworst(:,matType)),'--','linewidth',1.1);
% pl6 = semilogy(V,mean_ExSNR(:,matType)*bnd,'--k','linewidth',1.1);

legend([pl1; pl2; pl3; pl4; pl5],...
    {'Exhaustive Search', 'Greedy', 'Submodular Surrogate', ...
    'Convex Method','Worst Subset'},'location','sw')

xlabel('Number of Sensors [k]')
ylabel('Probability of Error')

%save('18-07-SmoothSnrComparisonRToeplitz') %0 seed