%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% 
% Goal: Evaluate the performance of DL-SBL
%
% Author: Geethu Joseph, ECE Dept., IISc
% Email : geethu@iisc.ac.in
% Date  : 10 Feb. 2017
% 
% This script/program is released under the Commons Creative Licence
% with Attribution Non-commercial Share Alike (by-nc-sa)
% http://creativecommons.org/licenses/by-nc-sa/3.0/
% Short Disclaimer: this script is for educational purpose only.
%
% 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [X, A_est] = DLSBL(Y, R,N, varargin)

%===============================================================================
%
%                               DLSBL Algorithm
%
% ===== INPUTS =====
%
%   Y           : m X L measurement matrix
%   R           : m X m noise covariance matrix
%
%===== OPTIONALS ======
%
%  'MAX_ITERS'   : Maximum number of iterations.
%                    [ Default value: MAX_ITERS = 2000 ]
%
%  'EPSILON'     : Threshold to stop the whole algorithm. 
%                    [ Default value: EPSILON = 1e-8   ]
%
%  'PRINT'       : Display flag. If = 1: show output; If = 0: supress output
%                    [ Default value: PRINT = 0        ]
%
% ===== OUTPUTS =====
%   X          : the estimated solution matrix, or called source matrix (size: M X L)
%   A           : m X N dictionary matrix
% *** Author ***
%   Geethu Joseph, ECE Dept., IISc
%   
% *** Version ***
%   1.0 (09/11/2016)
%
  
%% Dimension of the Problem
[m2 m3 L2] = size(R);
[m L] = size(Y);

% Dimesion Check:

if m2 ~= m3
    error('Covariance matrix should be square');
else 
    if m3 == 1
        R = R*eye(m);
    else 
        if m2 ~= m
            error('Dimensions of measurement vector and covariance should match');
        end;
    end;
end;

if L2 ~= 1 && L2 ~= L
      error('Dimensions of measurement vector and covarinace should match');
end;

%% Default Control Parameters
EPSILON     = 5*10^-3;        % threshold to stop
MAX_ITERS   = 1000000;         % maximum iterations
PRINT       = 0;            % don't show progress information

%% Initializations 

mu         = zeros(N,L);
% mu         = Xgen;
P          = zeros(N,N,L);
A          = Y(:,1:N);
A          = A./(ones(m,1)*sqrt(sum(A.^2)));
gamma      = ones(N,L);
% Control Flags

if(mod(length(varargin),2)==1)
    error('Optional parameters should always go by pairs\n');
else
    for i=1:2:(length(varargin)-1)
        switch lower(varargin{i})
            case 'epsilon'   
                EPSILON = varargin{i+1}; 
            case 'print'    
                PRINT = varargin{i+1}; 
            case 'max_iters'
                MAX_ITERS = varargin{i+1};
            case 'init'
                A = varargin{i+1};
            otherwise
                error(['DL-SBL: Unrecognized parameter: ''' varargin{i} '''']);
        end
    end
end



if (PRINT) fprintf('\nRunning DL-SBL ...\n'); end



for l = 1 : L
    P(:,:,l) = eye(N);
end
count      = 0;                        % iteration count
update_A   = 1;
%% *** Learning loop ***
while (1)

    mu_old  = mu;
    P_old   = P;
    A_old = A;
    gamma_old = gamma;
    
    Sigma = zeros(N,N);
    for l = 1 : L
        Gamma = diag(gamma(:,l));
        G     = diag(sqrt(gamma(:,l)));
        if norm(R) ~= 0
            J = Gamma*A'*(A*Gamma*A'+R)^-1;
        else
            J = G*pinv(A*G);
        end;
        
        mu(:,l) = J* Y(:,l);
        P(:,:,l)  = (eye(N) - J*A)*Gamma;
    
        
        temp = mu(:,l)*mu(:,l)'+P(:,:,l);
        gamma(:,l) = diag(temp);
        Sigma = Sigma + temp;
    end;
    Sigma = Sigma - diag(diag(Sigma));
    
    count_A = 1;
    YM = Y*mu';
    while (update_A)
        A_temp = A;
        for i = 1 : N
            temp = YM(:,i) - A*Sigma(:,i);
            if norm(temp) > 10^-5
                A(:,i) =  temp/norm(temp);
            end;
        end
               
        change = norm(A_temp-A,'fro')/norm(A_temp,'fro');
        if change < EPSILON break; end;
        count_A = count_A + 1;

        if count_A > MAX_ITERS*10 break; end;
    end

    count = count + 1;

    change_gamma = norm(gamma - gamma_old,'fro')/norm(gamma_old,'fro');
    change_A = norm(A-A_old,'fro')/norm(A_old,'fro');
    change = change_A+change_gamma;
    
    if (PRINT) 
    	disp(['iters: ',num2str(count),'   num coeffs: ',num2str(m), ...
                '   gamma change: ',num2str(change_gamma),...
                '   A change: ',num2str(change_A), '   A:count', num2str(count_A)]); 
	end;

	if (count >= MAX_ITERS)	break;  end;
    
	
    
	if (change < EPSILON)   break;  end;
    
    %if (change_A < EPSILON/10) update_A = 0; end;
end;


% Expand hyperparameters 

X = mu; 
A_est = A;
if (PRINT) fprintf('\nFinish running ...\n'); end
return;
