function [ data_ ] = uwacCSIDD( Zm, fc, A_p0, tau_p0, b_p0, data, T, M,...
    pilotCarriers, nullCarriers, pilotData, nullData,...
    mQAM, N0, structCSIDD)
% Perfect CSI based Data Detection (CSIDD)
% This function performs a data detection knowing the channel at the
% receiver. The following steps are executed:
%
% 1. Construct Channel Matrix: Construct the channel matrix as seen at each
% partial interval (FFT-) demodulator output. Then, depending on the
% measured data fed into the function (whether of size Kx1 or KxM), form
% the post-combined channel matrix. If the partial interval demodulator
% output is fed, use MMSE weights for linearly combining the outputs. Or
% else, simply sum up to get the channel matrix seen by full interval
% (FFT-) demodulator output.
%
% 2. MMSE estimate of data:
%
% 3. Demapping to constellation:
%
%
%INPUTS:
%   Zm        : Output data from either full interval (FFT-) demodulator OR
%             partial interval demodulator. For a full interval
%             demodulator, Zm is KxM where K is the total no. of
%             subcarriers (data + pilot +null). For a partial interval
%             demodulator, Zm is Kx1.
%
%   f         : vector of K sub-carrier frequencies
%
%   tau_p0    : Npaths x 1 vector of path delays
%
%   b_p0      : Npaths x 1 vector of scale factors due to doppler
%
%   data      : K x 1 vector of data symbols
%
%   T         : Duration of an OFDM symbol.
%
%   M         : Number of partial FFTs formed
%
%   structCSIDD: 
%
%OUTPUTS:
%  data_      : Kx1 vector of estimated data vector
%
%REFERENCE:
% [1] Arunkumar K.P. and Chandra R. Murthy, "Iterative Sparse Channel
% Estimation and Data Detection for Underwater Acoustic Communications
% Using Partial Interval Demodulation",  IEEE Transactions on Signal
% Processing ( Volume: 66 , Issue: 19 , Oct.1, 1 2018 )
%
%
%Author  : Arunkumar K. P.
%Address : Ph.D. Scholar,
%          Signal Processing for Communications Lab, ECE Department,
%          Indian Institute of Science, Bangalore, India-560 012.
%Email   : arunkumar@iisc.ac.in
%
%REVISION HISTORY
% Version : 2.1
% Last Revision: 05-02-2018
%
%
% This script/program is released under the Commons Creative Licence
% with Attribution Non-commercial Share Alike (by-nc-sa)
% http://creativecommons.org/licenses/by-nc-sa/3.0/
%
% Short Disclaimer: this script is for educational purpose only.
%++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%% Parameters
[K, ~] = size(Zm); % K = number of subcarriers

df = 1/T; %subcarrier spacing

f = fc + (-K/2:K/2-1)*df; %subcarriers

dataCarriers = sort( setdiff( 1:K, union( pilotCarriers, nullCarriers) ) ); % Ascertain the number of subcarriers & data subcarrier locations...

%% (1) Construct Channel matrix for the partial interval demodulator (PID)
[ HmEstim ] = uwacChannelMatrix(fc, K, T, M, A_p0, tau_p0, b_p0);
    %HmEstim is the channel matrix seen at partial interval demodulator output
    
Hfull = sum(HmEstim,3); %Hfull is one of the o/p arg (HmEstim is bulkier by a factor of M!)    
    %Hfull is the channel matrix seen at full interval demodulator output
    
if strcmpi(structCSIDD.DataDetection.Model,'FID')   
    % (a) Construct the full interval demodulator output
    Z = sum(Zm,2);
    
    % (b) Construct channel matrix for the full interval demodulator (FID)
    Hestim = Hfull; %channel matrix seen at full interval demodulator o/p    
    
    % (2) MMSE Estimation of data symbols
    data_ = zeros(K,1);    
    Z_ = Z - Hestim(:, pilotCarriers)*data(pilotCarriers);    
    Hestim_ = Hestim(:,dataCarriers);    
    data_(dataCarriers) = (Hestim_'*Hestim_+N0*eye(length(dataCarriers)))\(Hestim_'*Z_); %MMSE estimate of the data vector
    
    % (3) Demapping to M-QAM constellation
    data_(dataCarriers) = qammod( qamdemod(data_(dataCarriers),mQAM), mQAM );    
    data_(pilotCarriers) = pilotData;    
    data_(nullCarriers) = nullData;  %zero-out null data
    
else    
    % (a) Form optimum combiner weights for PID outputs
    [ Wopt ] = uwacOptimPartialFFTCombinerWts( f, tau_p0, b_p0, A_p0, T, M, N0  );
    
    % (b) Construct post-combined channel matrix...
    Wopt_H = repmat(Wopt, 1, 1, K); %Resize weights to match channel matrix dimensions    
    Wopt_H = permute(Wopt_H, [1 3 2]); %Reshape weights to match channel matrix dimensions    
    
    % (c) Construct combined channel matrix for the partial interval demodulator (PID)
    Hestim = sum(HmEstim.*conj(Wopt_H),3); % sum up PID channel matrices using combining Wts
        
    % (3.1d) Retain only diagonal terms of the post-combined channel matrix
    %         Hestim = diag( diag(Hestim) );%retain only diagonal entries
            
    % (e) Construct post-combined measurements
    Z = sum(Zm.*conj(Wopt),2);
    
    % (2) MMSE Estimation of data symbols
    data_ = zeros(K,1);    
    Z_ = Z - Hestim(:, pilotCarriers)*data(pilotCarriers);    
    Hestim_ = diag( Hestim(dataCarriers,dataCarriers) );
    data_(dataCarriers) = ( Hestim_.*Z_(dataCarriers) ) ./ ( abs(Hestim_).^2 + N0 );
    
    % (3) Demapping to M-QAM contellation
    data_(dataCarriers) = qammod( qamdemod(data_(dataCarriers),mQAM), mQAM );    
    data_(pilotCarriers) = pilotData;    
    data_(nullCarriers) = nullData;  %zero-out null data     
end

end
