function [ H0 ] = uwacChannelMatrix(fc, K, T, M, A_p0, tau_p0, b_p0, varargin)
%Channel Matix for UWAC
%   This function returns a K x K x M matrix H such that H(:,:,m) is
%   the (time-varying) channel matrix corresponding to partial FFT output
%   for m = 1,2,...,M. We may view sum(H,3) as the equivalent channel
%   matrix at the output of a full length FFT over OFDM duration.
%
%
%INPUTS:
%   fc     : carrier frequency
%
%   K      : Number of subcarriers
%
%   T      : Duration of an OFDM symbol.
%
%   M      : Number of partial FFTS formed
%
%OUTPUTS:
%   H      : K x K X M Channel Matrix
%
%
% REFERENCE:
% [1] Arunkumar K.P. and Chandra R. Murthy, "Iterative Sparse Channel
% Estimation and Data Detection for Underwater Acoustic Communications
% Using Partial Interval Demodulation",  IEEE Transactions on Signal
% Processing ( Volume: 66 , Issue: 19 , Oct.1, 1 2018 )
%
%
%Author  : Arunkumar K. P.
%Address : Ph.D. Scholar,
%          Signal Processing for Communications Lab, ECE Department,
%          Indian Institute of Science, Bangalore, India-560 012.
%Email   : arunkumar@iisc.ac.in
%
%REVISION HISTORY
%Version : 2.1
%Last Revision: 05-02-2018
%
%
% This script/program is released under the Commons Creative Licence
% with Attribution Non-commercial Share Alike (by-nc-sa)
% http://creativecommons.org/licenses/by-nc-sa/3.0/
%
% Short Disclaimer: this script is for educational purpose only.
%++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%% Initializations
df = 1/T; %subcarrier spacing

f = fc + (-K/2:K/2-1)*df; %subcarriers

dT = T/M; %time-step from one partial FFT output to the next

kMat = bsxfun(@plus, -(0:K-1)', 0:K-1); %integer matrix: [kMat]l,k = k-l

H0 = zeros(K, K, M); %Channel Matrices for M partial FFT outputs

for m = 1:M    
    %% Make partial FFT channel matrix
    tm = (m-1)*dT + dT/2; %time at the middle of partial FFT block
    
    tau_p_tm = tau_p0(:) - b_p0(:)*tm; %Time-varying delay within an OFDM symbol at the m'th partial FFT block: tau_p(t) = tau_p - a_p*t   
    
    Gmat = exp(-1i*2*pi*f(:)*tau_p_tm');
    
    Jm = 1/M * exp( 1i*2*pi*kMat*(2*m-1)/(2*M) ) .* sinc( kMat/M );
       
    H0(:,:,m) = Jm * diag(Gmat*A_p0(:,m)); %Channel Matrix as seen at m-th partial FFT output
    
end

if ~isempty(varargin)
    demodType = varargin{1};
    
    if strcmpi(demodType,'FID')%convert H0 to FID channel matrix if true
        H0 = sum(H0,3); %FID channel matrix
    end
end

end
