function [ A_p0, tau_p0, b_p0, H0 ] = uwacChannelParameters(T, K, M, Np, Tg, mu, bmax, varargin)
% UWA Channel Parameter Generation
%   This function draws amplitude, delay and Doppler spread parameters of
%   an underwater channel.
%
%
%INPUTS:
%   T      : Duration of an OFDM symbol.
%
%   K      : No. of Sub-carriers
%
%   M      : Number of partial FFTS formed
%
%   Np     : Number of propagation paths
%
%   Tg     : Guard Interval
%
%   mu     : Exponential PDF Parameter
%
%   bmax   : maximum doppler scale
%
%   lambda : Oversampling factor used for delay grids
%
%   Nb     : Number of grids used for Doppler-spread
%
%
%OUTPUTS:
%   Z      : K x M Received (Measured) Matrix (K = # of subcarriers, M = #
%            of measurement vectors = # of partial FFTs)
%
%   A0     : K x P x M Matrix in which A0(:,p,m) is the transfer function
%            of the p'th propagation path to the K outputs of the m'th
%            partial FFT.
%
%
%REFERENCE:
% [1] Arunkumar K.P. and Chandra R. Murthy, "Iterative Sparse Channel
% Estimation and Data Detection for Underwater Acoustic Communications
% Using Partial Interval Demodulation",  IEEE Transactions on Signal
% Processing ( Volume: 66 , Issue: 19 , Oct.1, 1 2018 )
%
%
%Author  : Arunkumar K. P.
%Address : Ph.D. Scholar,
%          Signal Processing for Communications Lab, ECE Department,
%          Indian Institute of Science, Bangalore, India-560 012.
%Email   : arunkumar@iisc.ac.in
%
%
%REVISION HISTORY
% Version : 2.1
% Last Revision: 05-02-2018
%
%
%
% This script/program is released under the Commons Creative Licence
% with Attribution Non-commercial Share Alike (by-nc-sa)
% http://creativecommons.org/licenses/by-nc-sa/3.0/
%
% Short Disclaimer: this script is for educational purpose only.
%++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%% generate arrival times
interArrivalTimes = exprnd(mu,Np,1);%inter arrival times ~ expon. with mean mu

interArrivalTimes = sort(interArrivalTimes);

tau_p0 = cumsum(interArrivalTimes);%arrival times at t=0

%% generate path attenuation coeffs
pathLoss_dB = tau_p0*(20/Tg);% Path attentuation (in dB)

pathLoss = 10.^(-pathLoss_dB/10);% Path attenuation (on linear scale)

A_p0 = diag( sqrt(pathLoss) )*randn(Np,M) ; %real path amplitudes: modified on 29/09/2017

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
A_p0 = repmat(A_p0(:,1),[1 M]); %ampl assumed const across partial FFTs
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% generate doppler scales
b_p0 = bmax*2*(rand(Np,1)-.5);

disp('Channel Parameters (A_p0, b_p0, tau_p0): CONTINUOUS (off-grid)');    

if ~isempty(varargin)
    
    fc = varargin{1};
    
    if length(varargin)>1
        demodType = varargin{2}; %'PID'/'FID'
    else
        demodType = 'PID';
    end
    
    % Compute the true channel matrix
    [ H0 ] = uwacChannelMatrix(fc, K, T, M, A_p0, tau_p0, b_p0, demodType);
    %H0 = PID channel matrix if demodType = 'PID', or else it is the FID
    %channel matrix
    
else
    
    H0 = [];
    
end

end
