function [ Z, A0, Vm ] = uwacChannelResponse(data, fc, T, M, A_p0, tau_p0, b_p0, N0, Tvm)
% UWA Channel Response Simulation
%   This function simulates the partial FFT output for m = 1, 2, ... , M at
%   the receiver of an UWA communication system when an OFDM data sequence
%   is transmitted from a transmitter. The transmitted OFDM symbol travels
%   along multiple paths, undergoes delay and attenuation, and experiences
%   time-scaling due to doppler.
%
%
%INPUTS:
%   data   : K x 1 vector of symbols (data + pilot + null)
%
%   fc     : carrier frequency
%
%   T      : Duration of an OFDM symbol.
%
%   M      : Number of partial FFTS formed
%
%   Np     : Number of propagation paths
%
%   N0     : Noise Power %Signal to Noise Ratio = 20*log10(||Hs||/||n||)
%
%
%OUTPUTS:
%   Z      : K x M Received (Measured) Matrix (K = # of subcarriers, M = #
%            of measurement vectors = # of partial FFTs)
%
%   A0     : K x P x M Matrix in which A0(:,p,m) is the transfer function
%            of the p'th propagation path to the K outputs of the m'th
%            partial FFT.
%
%
%REFERENCE:
% [1] Arunkumar K.P. and Chandra R. Murthy, "Iterative Sparse Channel
% Estimation and Data Detection for Underwater Acoustic Communications
% Using Partial Interval Demodulation",  IEEE Transactions on Signal
% Processing ( Volume: 66 , Issue: 19 , Oct.1, 1 2018 )
%
%
%Author  : Arunkumar K. P.
%Address : Ph.D. Scholar,
%          Signal Processing for Communications Lab, ECE Department,
%          Indian Institute of Science, Bangalore, India-560 012.
%Email   : arunkumar@iisc.ac.in
%
%
%REVISION HISTORY
% Version : 2.1
% Last Revision: 05-02-2018
%
%
% This script/program is released under the Commons Creative Licence
% with Attribution Non-commercial Share Alike (by-nc-sa)
% http://creativecommons.org/licenses/by-nc-sa/3.0/
%
% Short Disclaimer: this script is for educational purpose only.
%++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
K = length(data); %number of subcarriers

Np = length(tau_p0);

%% simulate channel response
df = 1/T; %subcarrier spacing

f = fc + (-K/2:K/2-1)*df; %subcarriers

Z = zeros(K,M);

dT = T/M; %time-step from one partial FFT output to the next

kMat = bsxfun(@plus, -(0:K-1)', 0:K-1); %integer matrix: [kMat]l,k = k-l

A0 = zeros(K, Np, M); %Measurement Matrices

Vm = zeros(K,M);

for m = 1:M    
    %% Make m'th partial FFT measurement matrix
    tm = (m-1)*dT + dT/2; %time at the middle of partial FFT block
    
    tau_p_tm = tau_p0(:) - b_p0(:)*tm; %Time-varying delay within an OFDM symbol at the m'th partial FFT block: tau_p(t) = tau_p - a_p*t   
    
    Gmat = exp(-1i*2*pi*f(:)*tau_p_tm');
    
    Jm = 1/M * exp( 1i*2*pi*kMat*(2*m-1)/(2*M) ) .* sinc( kMat/M );
    
    A0(:,:,m) = Jm * bsxfun(@times, Gmat, data(:)); % A(:,:,m) = Jm * diag(data) * Gmat;

    %% Generate (correlated) Noise vector    
    um = sqrt(1/2) * (randn(K,1) + 1i*randn(K,1)); %uncorrelated noise
    
    Vm(:,m) = sqrt(N0)*Tvm(:,:,m)*um; %correlated (colored) noise    

     %% Received Vector at m'th partial FFT output    
     Z(:,m) = A0(:,:,m)*A_p0(:,m) + Vm(:,m);  
end

end
