function [data, dataBits] = uwacDataSymbols( data, dataCarriers, mQAM, hTEnc )
%uwacDataSymbols 
%   This functions load the dataCarriers with M-QAM modulated version of
%   the coded/uncoded data bit stream
%
%
%INPUTS:
%   data      : Data vector whose locations corresponding to dataCarriers
%               (in the uncoded case) or dataCarriersUsed are loaded with
%               data symbols from an M-QAM constellation
%
%   dataCarriers : Indices of the data symbols
%
%   mQAM      : Order M of the M-QAM
%
%   hTEnc     : Handle to the turbo encorder object
%
%   dataCarriersUsed : Indices of the data symbols that are used in the
%                      coded case
%
%   frmLen     : Turbo code frame length
%
%   intrlvrIndices   : permutation used in the RSC of the turbo coder
%
%   hMod : Hnadle to the M-QAM modulator object
%
%
%OUTPUTS:
%   data     : K x 1 data vector loaded with randomly drawn data symbols
%
%   dataBits : data bit stream that are modulated to one of the M-QAM
%              symbol
%
%REFERENCE:
% [1] Arunkumar K.P. and Chandra R. Murthy, "Iterative Sparse Channel
% Estimation and Data Detection for Underwater Acoustic Communications
% Using Partial Interval Demodulation",  IEEE Transactions on Signal
% Processing ( Volume: 66 , Issue: 19 , Oct.1, 1 2018 )
%
%
%Author  : Arunkumar K. P.
%Address : Ph.D. Scholar,
%          Signal Processing for Communications Lab, ECE Department,
%          Indian Institute of Science, Bangalore, India-560 012.
%Email   : arunkumar@iisc.ac.in
%
%REVISION HISTORY
% Version : 2.1
% Last Revision: 05-02-2018
%
%
% This script/program is released under the Commons Creative Licence
% with Attribution Non-commercial Share Alike (by-nc-sa)
% http://creativecommons.org/licenses/by-nc-sa/3.0/
%
% Short Disclaimer: this script is for educational purpose only.
%++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%% Generate data vector...
if isempty(hTEnc) % Uncoded case
    % Generate random data symbols in {0,1,2,...,mQAM-1}
    dataBits = randi(mQAM,length(dataCarriers),1)-1;
    
    % Mount the modulated & encoded symbols on OFDM data subcarriers
    data(dataCarriers) = qammod(dataBits,mQAM);
    
else % Coded case
    error('Channel encoding and decoding are not implemented in this basic version');
end

end
