function [ tau_p, b_p ] = uwacDictionaryConfig( T, K, Tg, lambda, bmax, delBmax, varargin )
%uwacDictionaryConfig: Summary of this function goes here
%
% Dictionary grid setup: This script is invoked inline by uwacStart.m and
% uwacRun.m to set up the parameters of the dictionary matrix according to
% the Doppler scale and other CP-OFDM system parameters.
%
%INPUTS:
%   fc     : carrier frequency
%
%   K      : Number of subcarriers
%
%   T      : Duration of an OFDM symbol.
%
%   M      : Number of partial FFTS formed
%
%REFERENCE:
% [1] Arunkumar K.P. and Chandra R. Murthy, "Iterative Sparse Channel
% Estimation and Data Detection for Underwater Acoustic Communications
% Using Partial Interval Demodulation",  IEEE Transactions on Signal
% Processing ( Volume: 66 , Issue: 19 , Oct.1, 1 2018 )
%
%Author  : Arunkumar K. P.
%Address : Ph.D. Scholar,
%          Signal Processing for Communications Lab, ECE Department,
%          Indian Institute of Science, Bangalore, India-560 012.
%Email   : arunkumar@iisc.ac.in
%
%
%REVISION HISTORY
% Version : 2.1
% Last Revision: 05-02-2018
%
%
% This script/program is released under the Commons Creative Licence
% with Attribution Non-commercial Share Alike (by-nc-sa)
% http://creativecommons.org/licenses/by-nc-sa/3.0/
%
% Short Disclaimer: this script is for educational purpose only.
%++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%% Dictionary Parameters
tau_p = T/(lambda*K):T/(lambda*K):Tg; %delay grid: dTau = T/(lambda*K)

if (bmax ~=0 )
    bmax = max(bmax, delBmax);
    
    Nb = max(ceil( bmax / delBmax ),3); %# of doppler grids from -bmax to bmax
    
    b_p = -bmax: 2*bmax/(Nb-1): bmax;   %doppler grid when there is a doppler-spread
else
    b_p = 0;
end

%% Make CS dictionaries for Partial & Full FFT Demod. O/P
% done when varargin is nonempty
if ~isempty(varargin) %plot the dictionary matrix for a data with only pilots and rest set to ZERO
    
    fc = varargin{1};
    
    M = varargin{2};
    
    knownData = varargin{3};
    
    measuredCarriers = varargin{4};
    
    if length(varargin)>4
        hFigure = varargin{5}; %handle to the figure
    else
        hFigure = []; %plot in a new figure
    end
        
    df = 1/T; %subcarrier spacing
    
    f = fc + (-K/2:K/2-1)*df; %subcarriers
    
    % Make a CS dictionary matrix using known data (pilot + null data) and the
    % rest (on data subcarriers) set to zero
    [ Am_init ] = uwacMeasurementMatrix( f, tau_p, b_p, knownData, T, M );
    %Am_init is K x (Nt.Nb) x M matrix - Am_init(:,:,m) is the initial
    %dictionary corresponding to m'th partial FFT, constructed using only known
    %symbols while setting the unknown data symbols to zero.
    
    % Make a CS dictionary corresponding to data used for initial channel
    % estimation for each partial FFT output
    for m = 1:M
        Am_measured_init( (m-1)*length(measuredCarriers)+1 : m*length(measuredCarriers), : ) =...
            Am_init(measuredCarriers,:,m); %Am_measured_stack_init: Stacked up dictionaries of partial FFTs
    end
    
    % Make a CS dictionary (using known data) for full FFT output
    A_init = sum(Am_init,3); %Full FFT dictionary
    
    A_measured_init = A_init(measuredCarriers,:); %Full FFT dictionary for measured locations
    
    % Plot the normalized singular values
    figure(hFigure), cla,
    
    plot(svd(Am_measured_init)/max(svd(Am_measured_init)), 'ro'), grid on, hold on,
    
    plot(svd(A_measured_init)/max(svd(A_measured_init)), 'kp')
    
    xlabel('index of sorted singular values (k)'), ylabel('normalized  \sigma(k)')
    
end

end
