function [ data, ber ] = uwacIterativeDecoder(niters, structCEDD,...
    Zm, fc, tau_p, b_p, data, T, M,...
    measuredCarriers, pilotCarriers, nullCarriers, pilotData, nullData, mQAM, N0, structSparseAlgo,...
    dataCarriersUsed, hTDec, dataBits, H0, varargin)
% Iterative Decoding
%
% This function performs an iterative decoding of the (coded) data bits and
% outputs the estimated data symbols. Following steps are performed during
% each iteration:
%
% 1. Joint Channel Estimation and Data Detection: This is performed by
% calling the function 'uwacJointChanEstimAndDataDet.m'. The steps within
% uwacJointChanEstimAndDataDet.m are as follows:
%
%   (i) Construct the Dictionary Matrix: Using the tentative estimate of
%   data input to this function, and the grids of delays and doppler
%   spread, construct the dictionary at the output of each partial interval
%   demodulator
%
%   (ii). Sparse Channel Vector Recovery: Recover the channel vector either
%   from the full interval FFT-demodulator output OR the partial interval
%   FFT-demodulator output (decided based on the input measurements to this
%   function).
%
%   (iii) Estimate Channel Matrix: Estimate the channel matrix as seen at
%   each partial interval (FFT-) demodulator output. Then, depending on the
%   measured data fed into the function (whether of size Kx1 or KxM), form
%   the post-combined channel matrix. If the partial interval demodulator
%   output is fed, use MMSE weights for linearly combining the outputs. Or
%   else, simply sum up to get the channel matrix seen by full interval
%   (FFT-) demodulator output.
%
%   (iv) MMSE estimate of data
%
%   (v) Demapping to constellation
%
% 2. Demodulate the received signal
%
% 3. Decode the demodulated signal (using channel decoder corresponding to
% the channel encoder used at the input)
%
% 4. Compute Bit Error Rate and MSE (in channel matrix estimation)
%
% INPUTS:
%   niters    : Number of iterations to perform
%
%   structCEDD : a structure that contains fields mentioning the aaproach of
%              channel estimation and data detection
%
%   Zm        : Output data from either full interval (FFT-) demodulator OR
%             partial interval demodulator. For a full interval
%             demodulator, Zm is KxM where K is the total no. of
%             subcarriers (data + pilot +null). For a partial interval
%             demodulator, Zm is Kx1.
%
%   f         : vector of K sub-carrier frequencies
%
%   tau_p     : Npaths x 1 vector of path delays
%
%   b_p       : Ndopls x 1 vector of scale factors due to doppler
%
%   data      : K x 1 vector of data symbols
%
%   T         : Duration of an OFDM symbol.
%
%   M         : Number of partial FFTs formed
%
%   measuredCarriers : indices of the subcarriers that are used for channel
%                     estimation
%
%   pilotCarriers    : indices of the pilot subcarriers in the data vector
%
%   nullCarriers     : indices of the null subcarriers
%
%   pilotData        : pilot symbol
%
%   nullData         : null symbol (usually 0)
%
%   mQAM             : Order M of M-QAM
%
%   N0               : Noise variance
%
%   structSparseAlgo : structure related to the sparse recovery algorithm
%                      used
%
%   hDemod           : Handle to the demodulator object
%
%   dataCarriersUsed : indices/location of the data symbols that are coded
%
%   hTDec            : Handle to the channel decoder (turbo)
%
%   intrlvrIndices   : Interleaver (permutation) indices
%
%   dataBits         : True data bits
%
%   H0               : True (full interval demodulator) channel matrix
%
%   varargin{1}      : A_p0
%   varargin{2}      : tau_p0
%   varargin{3}      : b_p0
%
% OUTPUTS:
%  data      : Kx1 vector of the refined estimate of data vector
%
%REFERENCE:
% [1] Arunkumar K.P. and Chandra R. Murthy, "Iterative Sparse Channel
% Estimation and Data Detection for Underwater Acoustic Communications
% Using Partial Interval Demodulation",  IEEE Transactions on Signal
% Processing ( Volume: 66 , Issue: 19 , Oct.1, 1 2018 )
%
%Author  : Arunkumar K. P.
%Address : Ph.D. Scholar,
%          Signal Processing for Communications Lab, ECE Department,
%          Indian Institute of Science, Bangalore, India-560 012.
%Email   : arunkumar@iisc.ac.in
%
%REVISION HISTORY
% Version : 2.1
% Last Revision: 05-02-2018
%
%
% This script/program is released under the Commons Creative Licence
% with Attribution Non-commercial Share Alike (by-nc-sa)
% http://creativecommons.org/licenses/by-nc-sa/3.0/
%
% Short Disclaimer: this script is for educational purpose only.
%++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

for iter = 1:niters %iterate between channel estimation and data detection
    
    disp(['Iter = ' num2str(iter) ' (' structCEDD.Name ' using ' structSparseAlgo.Name ')']);
    
    switch(structCEDD.Name)
        case 'LS'            
            [data, H] = uwacLCEDD( Zm, pilotCarriers, pilotData, ...
                nullCarriers, nullData, mQAM);%LS works on Full FFT O/P
            
        case 'CSI'
            A_p0 = varargin{1};
            
            tau_p0 = varargin{2};
            
            b_p0 = varargin{3};
            
            H = H0; %CSI: knows the channel matrix perfectly
            
            [ data ] = uwacCSIDD( Zm, fc, A_p0, tau_p0, b_p0, data, T, M,...
                pilotCarriers, nullCarriers, pilotData, nullData,...
                mQAM, N0, structCEDD);
            % Function format:
            % [ data__ ] = uwacCSIDD( Zm, fc, A_p0, tau_p0, b_p0, data, T, M,...
            %    pilotCarriers, nullCarriers, pilotData, nullData,...
            %    mQAM, N0, structCSIDD);             
                        
        otherwise %default is sparse channel recovery
            [ data, ~, H ] = ...
                uwacJCEDD( Zm, fc, tau_p, b_p, data, T, M,...
                measuredCarriers, pilotCarriers, nullCarriers,...
                pilotData, nullData, mQAM, N0, structSparseAlgo, structCEDD, iter);
            % Function format:
            % [ data_, hRecover, HmEstim ] = uwacJCEDD( Zm, fc, tau_p, b_p, data, T, M,...
            %     measuredCarriers, pilotCarriers, nullCarriers, pilotData, nullData, ...
            %     mQAM, N0, structSparseAlgo, demodType, iCount)
    end
    
    % Channel decoding & error logging
    if isempty(hTDec) %No channel decoding
        % Demodulate the received signal
        demodSignal = qamdemod(data(dataCarriersUsed), mQAM);
        
        % Bit error rate (uncoded case)
        [~, ber] = biterr( dataBits, demodSignal);
        
    else
        error('Channel encoding and decoding are not implemented in this basic version');
    end    
    
end

end