function [data, H] = uwacLCEDD( Zm, pilotCarriers, pilotData, ...
    nullCarriers, nullData, mQAM)
%Least Squares Channel Estimation & Data Detection
% This function performs a LS based channel estimation and data detection.
% The following steps are performed-
%
% 1. Estimate the frequency-domain response of the channel corresponding to
% pilot carriers
%
% 2. Interpolate the channel response at pilot frequencies to obtain the
% responses at other frequencies
%
% 3. Estimate the data vector using the channel vector (in frequency
% domain) estimated above
%
% 4. Demodulate and obtain the data symbols
%
% INPUTS:
%   Zm             : KxM Output data from PID OR Kx1 Output from FID
%
%   pilotCarriers : (|S_P|x1) Location of pilot carriers in the Kx1 data vector
%
%   pilotData     : pilot symbols at pilot carrier locations
%
%   nullCarriers  : (|S_N|x1) Location of null carriers in the Kx1 data vector
%
%   nullData      : zero
%
%   mQAM          : Modulation order in M-QAM
%
% OUTPUTS:
%  data_      : Kx1 vector of refined estimate of data vector
%
%  hVec       : Diag(H), where H is the KxK channel matrix in the frequency
%             domain
%
%REFERENCE:
% [1] Arunkumar K.P. and Chandra R. Murthy, "Iterative Sparse Channel
% Estimation and Data Detection for Underwater Acoustic Communications
% Using Partial Interval Demodulation",  IEEE Transactions on Signal
% Processing ( Volume: 66 , Issue: 19 , Oct.1, 1 2018 )
%
%
%Author  : Arunkumar K. P.
%Address : Ph.D. Scholar,
%          Signal Processing for Communications Lab, ECE Department,
%          Indian Institute of Science, Bangalore, India-560 012.
%Email   : arunkumar@iisc.ac.in
%
%REVISION HISTORY
% Version : 2.1
% Last Revision: 05-02-2018

% This script/program is released under the Commons Creative Licence
% with Attribution Non-commercial Share Alike (by-nc-sa)
% http://creativecommons.org/licenses/by-nc-sa/3.0/
% Short Disclaimer: this script is for educational purpose only.
%++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%% Initializations
Z = sum(Zm,2); %FID output

K = length(Z);

data = zeros(K,1); %data vector

hVec = zeros(K,1); %channel vector (in freq dom)

interpCarriers = setdiff(1:K,pilotCarriers);

dataCarriers = setdiff(interpCarriers, nullCarriers);

%% STEP-1: Estimate the frequency-domain response of the channel
% corresponding to the pilot carriers
hVec(pilotCarriers) = Z(pilotCarriers)./pilotData; %Channel for pilot carriers

%% STEP-2: Interpolate the channel response from pilot frequencies 
% to obtain the responses at other frequencies
hVec(interpCarriers) = interp1(pilotCarriers, hVec(pilotCarriers),...
    interpCarriers, 'pchip');

H = diag(hVec); %Channel matrix: LS Channel Estimator assumes diagonal channel

%% STEP-3: Estimate the data vector using the channel vector 
% (in frequency domain) estimated above
data(pilotCarriers) = pilotData; %for pilot locations simply use pilot data

data(nullCarriers) = nullData; %for null locations simply use null data

data(dataCarriers) = Z(dataCarriers)./hVec(dataCarriers); %data estimation

%% STEP-4. Demodulate and obtain the data symbols
data(dataCarriers) = qammod( qamdemod(data(dataCarriers),mQAM), mQAM );