function [ A ] = uwacMeasurementMatrix( f, tau_p, b_p, data, T, M, varargin  )
%Measurement Matix for UWAC
%   This function generates a K x Ntau x M matrix A such that A(:,:,m) is
%   the measurement matrix corresponding to partial FFT output for m = 1,
%   2,...,M
%
%INPUTS:
%   f      : vector of K sub-carrier frequencies
%
%   tau_p  : Npaths x 1 vector of path delays
%
%   a_p    : Ndopls x 1 vector of scale factors due to doppler
%
%   data   : K x 1 vector of data symbols which are all zeros except at
%            pilot locations.
%
%   T      : Duration of an OFDM symbol.
%
%   M      : Number of partial FFTs formed
%
%   varargin{1}: iCount => if (icount>1) then use change in data to update A
%OUTPUTS:
%   A      : Kcar x Ntau X M Measurement Matrices
%
%REFERENCE:
% [1] Arunkumar K.P. and Chandra R. Murthy, "Iterative Sparse Channel
% Estimation and Data Detection for Underwater Acoustic Communications
% Using Partial Interval Demodulation",  IEEE Transactions on Signal
% Processing ( Volume: 66 , Issue: 19 , Oct.1, 1 2018 )
%
%Author  : Arunkumar K. P.
%Address : Ph.D. Scholar,
%          Signal Processing for Communications Lab, ECE Department,
%          Indian Institute of Science, Bangalore, India-560 012.
%Email   : arunkumar@iisc.ac.in
%
%REVISION HISTORY
% Version : 2.1
% Last Revision: 05-02-2018
%
%
% This script/program is released under the Commons Creative Licence
% with Attribution Non-commercial Share Alike (by-nc-sa)
% http://creativecommons.org/licenses/by-nc-sa/3.0/
%
% Short Disclaimer: this script is for educational purpose only.
%++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
persistent Gmat Jm prevData prevA

K = length(f); %No. of sub-carriers in an OFDM symbol

Npaths = length(tau_p); %No. of wave propagation paths (some paths may not be active when A is a dictionary)

Ndopls = length(b_p); %No. of doppler scales

Ngrids = Npaths*Ndopls; %No. of columns of Am for each m = 1:M, corresponds to # of parameters used to form the dictionary

if isempty(Gmat)
    
    dT = T/M; %time-step from one partial FFT output to the next
    
    tau_p_t = zeros(Ngrids,M);
    
    kMat = bsxfun(@plus, -(0:K-1)', 0:K-1); %integer matrix: [kMat]l,k = k-l
    
    Gmat = single( zeros(K, Ngrids, M) );
    
    Jm = single( zeros(K, K, M) );
    
    A = single( zeros(K, Ngrids, M) ); %Measurement Matrices
    
    for m = 1:M
        
        tm = (m-1)*dT + dT/2; %time at the middle of partial FFT block
        
        tau_p_tm = bsxfun(@minus, tau_p(:), b_p(:)'*tm); %Time-varying delay within an OFDM symbol at the m'th partial FFT block: tau_p(t) = tau_p - a_p*t
        
        tau_p_t(:,m) = tau_p_tm(:);%all delay parameters vectorized
        
        Gmat(:,:,m) = exp(-1i*2*pi*f(:)*tau_p_t(:,m)');
        
        Jm(:,:,m) = 1/M * exp( 1i*2*pi*kMat*(2*m-1)/(2*M) ) .* sinc( kMat/M );
        
        A(:,:,m) = Jm(:,:,m) * diag(data) * Gmat(:,:,m); %Jm(:,:,m)*bsxfun(@times, Gmat(:,:,m), data(:));
        
    end;
    
    prevData = data;
    
    prevA = A;
    
else
    
    if isempty(varargin)
        
        iCount = 1;
        
    else
        
        iCount = varargin{1};
        
    end;
    
    if iCount==1
        
        A = single( zeros(K, Ngrids, M) ); %Measurement Matrices
        
        for m = 1:M
            
            A(:,:,m) = Jm(:,:,m) * diag(data) * Gmat(:,:,m); %Jm(:,:,m)*bsxfun(@times, Gmat(:,:,m), data(:));
            
        end
        
    else %update A using only change in the 'data'
        
        dData = data(:) - prevData(:); %change in the data symbols
        
        dA = single( zeros(K, Ngrids, M) ); %Measurement Matrices
        
        for m = 1:M
            
            dA(:,:,m) = Jm(:,dData~=0,m) * diag(nonzeros(dData)) * Gmat(dData~=0,:,m); %Jm(:,:,m)*bsxfun(@times, Gmat(:,:,m), data(:));
            
        end        
        
        A = prevA + dA;
        
    end
    
    prevData = data;
    
    prevA = A;
    
end
