function [ Wopt ] = uwacOptimPartialFFTCombinerWts( f, tau_p0, a_p0, A_p0, T, M, N0  )
%Optimum (MMSE) Combiner Weights for Partial FFT Outputs
%   This function returns an M x K matrix W whole columns are
%
%INPUTS:
%   f      : vector of K sub-carrier frequencies
%
%   tau_p0 : P x 1 vector of propagation delays for each path
%
%   a_p0   : P x 1 vector of scale factors due to doppler for each path
%
%   A_p0   : P x 1 vector of channel (complex) amplitudes for each path
%
%   T      : Duration of an OFDM symbol.
%
%   M      : Number of partial FFTs formed
%
%   N0     : Noise level
%
%OUTPUTS:
%   Wopt   : K x M Weight Matrix
%
%REFERENCE:
% [1] Arunkumar K.P. and Chandra R. Murthy, "Iterative Sparse Channel
% Estimation and Data Detection for Underwater Acoustic Communications
% Using Partial Interval Demodulation",  IEEE Transactions on Signal
% Processing ( Volume: 66 , Issue: 19 , Oct.1, 1 2018 )
%
%Author  : Arunkumar K. P.
%Address : Ph.D. Scholar,
%          Signal Processing for Communications Lab, ECE Department,
%          Indian Institute of Science, Bangalore, India-560 012.
%Email   : arunkumar@iisc.ac.in
%
%REVISION HISTORY
% Version : 2.1
% Last Revision: 05-02-2018
%
% This script/program is released under the Commons Creative Licence
% with Attribution Non-commercial Share Alike (by-nc-sa)
% http://creativecommons.org/licenses/by-nc-sa/3.0/
%
% Short Disclaimer: this script is for educational purpose only.
%++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
K = length(f); %No. of sub-carriers in an OFDM symbol

dT = T/M; %time-step from one partial FFT output to the next

%% Construct optimum weight vectors for each subcarriers
Wopt = zeros(K,M);

tm = (0:M-1)'*dT + dT/2; %time at the middle of partial FFT block

tau_p0_tm = bsxfun(@minus, tau_p0(:)', tm*a_p0(:)'); %Time-varying delay within an OFDM symbol at the m'th partial FFT block: tau_p(t) = tau_p - a_p*t

Hl = zeros(K,M);

for l = 1 : K
    
    Hl(l,:) = diag( exp(-1i*2*pi*f(l)*tau_p0_tm) * A_p0 );
          
end

l = (1:K)';

m = 1:M;

for k = 1 : K    
   
    Vlk = 1/M * bsxfun(@times, exp( 1i*2*pi*(l-k)*(2*m-1)/(2*M) ), sinc( (l-k)/M ));
    
    Hlk = (Hl.*Vlk).'; % MxK Matrix whose k'th column is (Hl((k,:).*Vlk(k,:))." = Hk*v0
    
    Ryk = Hlk * Hlk' + N0/M *eye(M);
    
    Wopt(k,:) = (Ryk \ Hlk(:,k)).';
    
    Wopt(k,:) = Wopt(k,:)/norm(Wopt(k,:),2)*sqrt(M);
end

end