function [ Tvm, Cm ] = uwacPfftDemodCorrNoiseGenTfm(K, M, N0)
% Correlated Noise Generating Transform (used at Partial FFT Output)
%   This function constructs the matrix transform that when applied on
%   circular complex Gaussian noise vector will generate correlated complex
%   Gaussian noise vector at the partial FFT output
%
%
%INPUTS:
%   K      : Number of subcarriers
%
%   M      : Number of partial FFTS formed
%
%   N0     : Noise Power %Signal to Noise Ratio = 20*log10(||Hs||/||n||)
%
%
%OUTPUTS:
%   Tvm    : KxKxM transform matrix for generating correlated noise at the
%   output of the partial interval demodulator
%
%REFERENCE:
% [1] Arunkumar K.P. and Chandra R. Murthy, "Iterative Sparse Channel
% Estimation and Data Detection for Underwater Acoustic Communications
% Using Partial Interval Demodulation",  IEEE Transactions on Signal
% Processing ( Volume: 66 , Issue: 19 , Oct.1, 1 2018 )
%
%
%Author  : Arunkumar K. P.
%Address : Ph.D. Scholar,
%          Signal Processing for Communications Lab, ECE Department,
%          Indian Institute of Science, Bangalore, India-560 012.
%Email   : arunkumar@iisc.ac.in
%
%REVISION HISTORY
% Version : 2.1
% Last Revision: 05-02-2018
%
%
% This script/program is released under the Commons Creative Licence
% with Attribution Non-commercial Share Alike (by-nc-sa)
% http://creativecommons.org/licenses/by-nc-sa/3.0/
%
% Short Disclaimer: this script is for educational purpose only.
%++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

% Initializations...
kMat = bsxfun(@plus, -(0:K-1)', 0:K-1); %integer matrix: [kMat]l,k = k-l

Cm = zeros(K, K, M); %Noise covariance matrix for each m

Tvm = zeros(K, K, M); %Noise covariance matrix for each m

%% Build Transform Matrix/Filter that generates correlated Noise vector  
for m = 1:M    
    
    Cm(:,:,m) = N0/M * exp( 1i*2*pi*kMat*(2*m-1)/(2*M) ) .* sinc( kMat/M ); %(colored, rank-defficient) noise covariance

    [Uvm, Dvm] = eig(Cm(:,:,m));
    
    Tvm(:,:,m) = Uvm*sqrt(abs(real(Dvm)));     
end

end
