% THIS PROGRAM CONTAINS THE CORE LOOP FOR MONTE_CARLO RUNS IN uwacStart.m
%
% The program simulates an underwater channel based on a ray-theoretic
% model of sound propagation for analyzing the performance of an OFDM
% receiver employing partial FFT demodulation scheme. The ray-theoretic
% model expresses the signal at a receiver located underwater as a sum of a
% few scaled and delayed versions of the waveform transmitted from a source
% located away from the receiver. The amplitude and delay depends on the
% propagation path of the sound ray travelling from source to the receiver
% in a given underwater environment. Depending on the acoustics of an
% underwater propagation environment, sound rays travel along multiple
% paths and their amplitude-loss and delay along each path vary with time.
% Within the duration of one OFDM symbol, the variation in amplitude may be
% negligible. However, path delays could vary due to relative motion
% between the source and receiver. The effect of time-varing path delay,
% due to constant relative velocity, is to time-scale the waveform
% travelling along each path. The time-scale is path dependent, and in some
% severe environments a significant spread in their values across multiple
% paths is likely.
%
% A two-stage algorithm proposed in [1] is implemented in this script. The
% first stage uses pilot only observations at the output of partial FFT
% demodulator to estimate the channel, detect the data and then iterate.
% The second stage uses a full FFT demodulator and then iterates between
% channel estimation and data detection starting with the data detected at
% the end of stage-1 iterations. Notice that at stage of iteraton, the CS
% dictionary for channel estimation is refined using the data detected in
% the previous iteration.
%
%REFERENCE:
% [1] Arunkumar K.P. and Chandra R. Murthy, "Iterative Sparse Channel
% Estimation and Data Detection for Underwater Acoustic Communications
% Using Partial Interval Demodulation",  IEEE Transactions on Signal
% Processing ( Volume: 66 , Issue: 19 , Oct.1, 1 2018 )
%
%
%Author  : Arunkumar K. P.
%Address : Ph.D. Scholar,
%          Signal Processing for Communications Lab, ECE Department,
%          Indian Institute of Science, Bangalore, India-560 012.
%Email   : arunkumar@iisc.ac.in
%
%
%REVISION HISTORY
% Version : 2.1
% Last Revision: 05-02-2018
%
%
% This script/program is released under the Commons Creative Licence
% with Attribution Non-commercial Share Alike (by-nc-sa)
% http://creativecommons.org/licenses/by-nc-sa/3.0/
%
% Short Disclaimer: this script is for educational purpose only.
%++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%% 1. Prepare for the Monte Carlo Runs
%<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
disp(['Preparing for ' num2str(ntrials-nstart+1) ' Monte Carlo runs...'...
    'nstart = ' num2str(nstart) ', nend = ' num2str(ntrials)])

% (1a) Compute the UWA Channel Statistics
%To estimate the power of the signal part of the received vector
rPow = zeros(size(bmax)); %E{||Hs||^2} = power of the received signal part at the end of the UWA channel

dPow = zeros(size(bmax)); %E{||s||^2} = power of the transmitted data/signal vector

hPow = zeros(size(bmax)); %E{tr(H'*H)} = channel loss

for iBmax = 1:length(bmax)
    [rPow(iBmax), dPow(iBmax)] = ...
        uwacStats(fc, K, T, M, Npath, Tg, mu, bmax(iBmax), mQAM, knownCarriers, knownData, 1000);%, structChannel);
    
    hPow(iBmax) = rPow(iBmax)/dPow(iBmax)*K;
    
    % Display the power of the signal components in the received vector
    disp(['Mean power of the received signal: E{||Hs||^2} ~= '...
        num2str(rPow(iBmax)) ' (Estimated from 1000 runs, bmax=' num2str(bmax(iBmax)) ')']);
    
    disp(['Mean power of the channel: E{tr(H''H)} ~= '...
        num2str(hPow(iBmax)) ' (Estimated from 1000 runs, bmax=' num2str(bmax(iBmax)) ')']);
end

% (1b) Construct transforming matrix to generate correl noise at partial FFT o/p
[ Tvm ] = uwacPfftDemodCorrNoiseGenTfm(K, M, 1);

disp('Done...');
%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

%% 2. Monte Carlo Runs...
%<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
figure(5); pause(.01); %Shows BER plot

for trial = nstart:ntrials % loop over ntrials: for each trial, results for all snr are generated
    
    tTrial = tic; % timer for one trial run
    
    %% (2.1) Loop across the snr variable...
    %% Entering the snr loop
    Bmax = bmax; %Doppler scale parameter & pilot arrangement are fixed
    
    %         L = pilotSpacing;
    
    %% Generate the data vector (indep. of Doppler and SNR) ...
    [data, dataBits] = uwacDataSymbols( data, dataCarriers, mQAM, hTEnc );
    
    %% Generate Channel Parameters (indep. of SNR and Pilot arrangement)       
    [ A_p0, tau_p0, b_p0, H0 ] = ...
        uwacChannelParameters(T, K, M, Npath, Tg, mu, bmax, fc, 'FID');
    
    %% Monte-Carlo Runs across the snr variable...
    %<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
    uwacSnrRun; %uwacSnrRun.m carries out Monte-Carlo Runs for various SNRs
    %>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>    
    
    %% (2c) Update plot
    set(0, 'CurrentFigure', 5); cla,  uwacPlotBER;
    title(['BER vs ' xAxisTitle...
        ' (Trial ' num2str(trial) ' of ' num2str(ntrials)...
        ': ' num2str(toc(tTrial)) ' sec/trial)'])
    if metaflag,
        savefig('uwacBER.fig');
    end
        
    pause(.001);
    
    %% (2d) Save metadata if the metaflag is true
    if metaflag
        %write to metafile...
        disp('Saving meta data...');
        fileID = fopen(metafile,'w');
        fprintf(fileID, ['Trial ' num2str(trial) ' of ' num2str(ntrials) ' trials \n']);
        fprintf(fileID, '%\n', ['Time taken : ' num2str(toc(tTrial)) ' seconds \n']);
        fprintf(fileID, metavars);
        fprintf(fileID, '\n');
        fprintf(fileID, metaformat, metadata);
        fclose(fileID);
        
        %reset all meta data after saving...
        metavars = loopType;
        metaformat = '%2.4f';
        metadata = loopVar(:);
    end
    
    %% (2e) Display execution time
    disp(['Time taken : ' num2str(toc(tTrial))...
        ' seconds. (Trial '...
        num2str(trial) ' of ' num2str(ntrials) ')']),
    
    %% (2f) Save results once during the time specified by saveRate
    saveTime = saveTime + toc(tTrial)/60; %in minutes
    if saveTime > saveRate %save results when saveTime exceeds saveRate
        disp('Saving all variables...');
        save(resultFileName);
        saveTime = 0;        %reset the savetime after saving...
    end
    
end
%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

%% 3. Plot final results
%<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
berPlotFigureFileName = ['uwacBER-' 'D' num2str(D) '-M' num2str(M)...
    '-Run' num2str(trial)...
    '-' structOMP.Name num2str(structOMP.Param)...
    '-' datestr(now,30) '.fig']; %BER plots are saved to this file

hBerPlots = figure; uwacPlotBER;

title(['BER vs ' xAxisTitle])

savefig(hBerPlots, berPlotFigureFileName);

%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

%% 4. Save results
%<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
save(resultFileName); %saves all the program variable here
%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

% exit;