% THIS PROGRAM CONTAINS THE CORE LOOP FOR THE MONTE_CARLO RUNS ACROSS SNR
%
%
% The program simulates an underwater channel based on a ray-theoretic
% model of sound propagation for analyzing the performance of an OFDM
% receiver employing partial FFT demodulation scheme. The ray-theoretic
% model expresses the signal at a receiver located underwater as a sum of a
% few scaled and delayed versions of the waveform transmitted from a source
% located away from the receiver. The amplitude and delay depends on the
% propagation path of the sound ray travelling from source to the receiver
% in a given underwater environment. Depending on the acoustics of an
% underwater propagation environment, sound rays travel along multiple
% paths and their amplitude-loss and delay along each path vary with time.
% Within the duration of one OFDM symbol, the variation in amplitude may be
% negligible. However, path delays could vary due to relative motion
% between the source and receiver. The effect of time-varing path delay,
% due to constant relative velocity, is to time-scale the waveform
% travelling along each path. The time-scale is path dependent, and in some
% severe environments a significant spread in their values across multiple
% paths is likely.
%
% A two-stage algorithm proposed in [1] is implemented in this script. The
% first stage uses pilot only observations at the output of partial FFT
% demodulator to estimate the channel, detect the data and then iterate.
% The second stage uses a full FFT demodulator and then iterates between
% channel estimation and data detection starting with the data detected at
% the end of stage-1 iterations. Notice that at stage of iteraton, the CS
% dictionary for channel estimation is refined using the data detected in
% the previous iteration.
%
%
%REFERENCE:
% [1] Arunkumar K. P. and Chandra R. Murthy, "Iterative Sparse Channel
% Estimation and Data Detection for Underwater Acoustic Communications
% Using Partial Interval Demodulation",  IEEE Transactions on Signal
% Processing ( Volume: 66 , Issue: 19 , Oct.1, 1 2018 )
%
%
%
%Author  : Arunkumar K. P.
%Address : Ph.D. Scholar,
%          Signal Processing for Communications Lab, ECE Department,
%          Indian Institute of Science, Bangalore, India-560 012.
%Email   : arunkumar@iisc.ac.in
%
%
%REVISION HISTORY
% Version : 2.1
% Last Revision: 05-02-2018
%
%
%
% This script/program is released under the Commons Creative Licence
% with Attribution Non-commercial Share Alike (by-nc-sa)
% http://creativecommons.org/licenses/by-nc-sa/3.0/
%
% Short Disclaimer: this script is for educational purpose only.
%++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%% 1. Initialize the local/temporary BER & MSE buffers for the current trial
ber_OMP_Stage1_partFftPilot_trial = zeros(length(snr),1);

ber_OMP_Stage2_fullFft_trial = zeros(length(snr),1);

ber_CS_fullFftPilot_trial = zeros(length(snr),1);

ber_CS_fullFft_trial = zeros(length(snr),1);

ber_LS_trial = zeros(length(snr),1); %BER of Least Squares data detection algo.

ber_CSI_trial = zeros(length(snr),1); %BER of genie aided data detector

%% 2. BER & MSE performance of various algorithms - for comparison
parfor iSnr = 1:length(snr) % loop over all snr values in the range
    
    tSnr  = tic; % timer for one snr result generation
    
    SNR = snr(iSnr); % signal to noise ratio
    
    N0 = rPow/K * 10^(-SNR/10); %Noise Power s.t. SNR is as required by the user
    
    disp(['SNR=' num2str(SNR) 'dB, '...
        'Trial ' num2str(trial) ' of ' num2str(ntrials) ' '...
        '(N0=' num2str(N0) ', '...
        'E{||Hs||^2}=' num2str(rPow) ', '...
        'E{tr(H''H)}=' num2str(hPow) ')']);
    
    Hpow = hPow;
    
    %% Simulate UWAC Channel Response
    [ Zm, A0 ] = uwacChannelResponse(data, fc, T, M, A_p0, tau_p0, b_p0, N0, Tvm);
    
    %% <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
    %% STAGE-1: OMP Recovery from Partial FFT O/P for only Pilot Subcarriers
    % Use data measured at partial FFT output to estimate the channel
    % and hence detect the data using pilot only observations; start
    % with known data on plot and null subcarriers
    if ~isempty(regexpi(algoList,'omp-stage-1','IgnoreCase')) || ~isempty(regexpi(algoList,'omp-stage-2','IgnoreCase'))
        [ data_OMP_Stage1_partFftPilot, ber ] =...
            uwacIterativeDecoder(3, structJCEDD_Stage_1,...
            Zm, fc, tau_p, b_p, knownData, T, M,...
            measuredCarriers, pilotCarriers, nullCarriers, pilotData, nullData, mQAM, N0, structOMP,...
            dataCarriersUsed, hTDec, dataBits, H0);
        % Function Format:
        % [ data, ber, mse ] = uwacIterativeDecoder(niters, stageName,...
        %     Zm, fc, tau_p, b_p, data, T, M,...
        %     measuredCarriers, pilotCarriers, nullCarriers, pilotData, nullData, mQAM, N0, structSparseAlgo,...
        %     dataCarriersUsed, hTDec, H0)
        
        ber_OMP_Stage1_partFftPilot_trial(iSnr) = ber;
    end
    
    %% STAGE-2: OMP Recovery from Full FFT O/P using entire observation (data + pilot + null)
    % Use data measured at Full FFT demodulator output to estimate the
    % channel and hence detect the data using entire observations;
    % start with data detected at the end of stage-1
    if ~isempty(regexpi(algoList,'omp-stage-2','IgnoreCase'))
        [ data_OMP_Stage2_fullFft, ber ] =...
            uwacIterativeDecoder(3, structJCEDD_Stage_2,...
            Zm, fc, tau_p, b_p, data_OMP_Stage1_partFftPilot, T, M,...
            1:K, pilotCarriers, nullCarriers, pilotData, nullData, mQAM, N0, structOMP,...
            dataCarriersUsed, hTDec, dataBits, H0);
        % Function Format:
        % [ data, ber, mse ] = uwacIterativeDecoder(niters, stageName,...
        %     Zm, fc, tau_p, b_p, data, T, M,...
        %     measuredCarriers, pilotCarriers, nullCarriers, pilotData, nullData, mQAM, N0, structSparseAlgo,...
        %     dataCarriersUsed, hTDec, H0)
        
        ber_OMP_Stage2_fullFft_trial(iSnr) = ber;        
    end
    %>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
        
    %% <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
    %% CS Recovery from Full-FFT O/P for Pilot Subcarriers - for performance comparison
    % Christian Berger et. al. (1)
    if ~isempty(regexpi(algoList,'berger','IgnoreCase')) || ~isempty(regexpi(algoList,'huang','IgnoreCase'))
        [ data_CS_fullFftPilot, ber ] =...
            uwacIterativeDecoder(3, structBerger,...
            Zm, fc, tau_p, b_p, knownData, T, M,...
            measuredCarriers, pilotCarriers, nullCarriers, pilotData, nullData, mQAM, N0, structOMP,...
            dataCarriersUsed, hTDec, dataBits, H0);
        % Function Format:
        % [ data, ber, mse ] = uwacIterativeDecoder(niters, stageName,...
        %     Zm, fc, tau_p, b_p, data, T, M,...
        %     measuredCarriers, pilotCarriers, nullCarriers, pilotData, nullData, mQAM, N0, structSparseAlgo,...
        %     dataCarriersUsed, hTDec, H0)
        
        ber_CS_fullFftPilot_trial(iSnr) = ber;        
    end
    %>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
        
    %% <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
    %% LS channel estimation & signal detection - for performance comparison
    if ~isempty(regexpi(algoList,'ls','IgnoreCase'))
        [ data_LS, ber ] =...
            uwacIterativeDecoder(1, structLCEDD,...
            Zm, fc, tau_p, b_p, knownData, T, M,...
            measuredCarriers, pilotCarriers, nullCarriers, pilotData, nullData, mQAM, N0, structOMP,...
            dataCarriersUsed, hTDec, dataBits, H0);
        % Function Format:
        % [ data, ber, mse ] = uwacIterativeDecoder(niters, stageName,...
        %     Zm, fc, tau_p, b_p, data, T, M,...
        %     measuredCarriers, pilotCarriers, nullCarriers, pilotData, nullData, mQAM, N0, structSparseAlgo,...
        %     dataCarriersUsed, hTDec, H0)
        
        ber_LS_trial(iSnr) = ber;       
    end
    %>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    
    %% <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
    %% CSI-aware Data Decoding (Genie)
    if ~isempty(regexpi(algoList,'csi','IgnoreCase'))
        [ data_CSI, ber ] =...
            uwacIterativeDecoder(1, structCSI,...
            Zm, fc, tau_p, b_p, knownData, T, M,...
            measuredCarriers, pilotCarriers, nullCarriers, pilotData, nullData, mQAM, N0, structOMP,...
            dataCarriersUsed, hTDec, ...
            dataBits, H0, A_p0, tau_p0, b_p0);
        % Function Format:
        % [ data, ber, mse ] = uwacIterativeDecoder(niters, stageName,...
        %     Zm, fc, tau_p, b_p, data, T, M,...
        %     measuredCarriers, pilotCarriers, nullCarriers, pilotData, nullData, mQAM, N0, structSparseAlgo,...
        %     dataCarriersUsed, hTDec, H0)
        
        ber_CSI_trial(iSnr) = ber;
    end
    %>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    
    %% Display execution time
    disp(['Time taken : ' num2str(toc(tSnr))...
        ' (SNR ' num2str(SNR) 'dB, Trial '...
        num2str(trial) ' of ' num2str(ntrials) ')']),    
end

%% 3. Update the global BER buffers for the current trial
ber_OMP_Stage1_partFftPilot(trial,:,:) = ber_OMP_Stage1_partFftPilot_trial;

ber_OMP_Stage2_fullFft(trial,:,:) = ber_OMP_Stage2_fullFft_trial;

ber_CS_fullFftPilot(trial,:,:) = ber_CS_fullFftPilot_trial;

ber_CS_fullFft(trial,:,:) = ber_CS_fullFft_trial;

ber_LS(trial,:,:) = ber_LS_trial; %BER of Least Squares data detection algo.

ber_CSI(trial,:) = ber_CSI_trial; %BER of genie aided data detector
