% THIS IS THE CODE TO RUN FOR EVALUATING BER PERFORMANCE OF PID RECEIVER
%
% Run this program to start demo of the algorithm in the paper [1]. After
% setting up parameters of simulation and the buffers for storing results,
% this code calls the script uwacRun.m which in turn carries out the Monte
% Carlo runs.
%
% Description:
% This program simulates an underwater channel based on a ray-theoretic
% model of sound propagation for analyzing the performance of an OFDM
% receiver employing partial FFT demodulation scheme. The ray-theoretic
% model expresses the signal at a receiver located underwater as a sum of a
% few scaled and delayed versions of the waveform transmitted from a source
% located away from the receiver. The amplitude and delay depends on the
% propagation path of the sound ray travelling from source to the receiver
% in a given underwater environment. Depending on the acoustics of an
% underwater propagation environment, sound rays travel along multiple
% paths and their amplitude-loss and delay along each path vary with time.
% Within the duration of one OFDM symbol, the variation in amplitude may be
% negligible. However, path delays could vary due to relative motion
% between the source and receiver. The effect of time-varing path delay,
% due to constant relative velocity, is to time-scale the waveform
% travelling along each path. The time-scale is path dependent and in some
% severe environments a significant spread in their values across multiple
% paths is likely.
%
% A two-stage algorithm proposed in [1] is implemented in this demo script.
% The first stage uses pilot only observations at the output of partial FFT
% demodulator to estimate the channel, detect the data and then iterate.
% The second stage uses a full FFT demodulator and then iterates between
% channel estimation and data detection starting with the data detected at
% the end of stage-1 iterations. Notice that at stage of iteraton, the CS
% dictionary for channel estimation is refined using the data detected in
% the previous iteration.
%
%REFERENCE:
% [1] Arunkumar K.P. and Chandra R. Murthy, "Iterative Sparse Channel
% Estimation and Data Detection for Underwater Acoustic Communications
% Using Partial Interval Demodulation",  IEEE Transactions on Signal
% Processing ( Volume: 66 , Issue: 19 , Oct.1, 1 2018 )
%
%
%Author  : Arunkumar K. P.
%Address : Ph.D. Scholar,
%          Signal Processing for Communications Lab, ECE Department,
%          Indian Institute of Science, Bangalore, India-560 012.
%Email   : arunkumar@iisc.ac.in
%
%REVISION HISTORY
% Version : 2.1
% Last Revision: 05-02-2018
%
% This script/program is released under the Commons Creative Licence
% with Attribution Non-commercial Share Alike (by-nc-sa)
% http://creativecommons.org/licenses/by-nc-sa/3.0/
%
% Short Disclaimer: this script is for educational purpose only.
%++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

clear all; %"clear all" is needed because there are persistent variables in this code

clc;

defpaper;

close all;

warning off;

%% 0. Basic settings for the Monte Carlo Runs
%<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
ntrials = 1000; %number of Monte Carlo runs = number of channel instantiations

snr = 0:3:15; %signal to noise ratio (see paper for definition)

bmax = 10e-4; %maximum doppler spread: the time-scaling parameter (due to Doppler) of
    %each propagating path 'p', ap = a + bp, where bp lies in [-bmax bmax]

metaflag = false; %If "true" it saves additional data during the run.

algoList = 'csi, ls, berger, omp-stage-2'; 

%% 1. CP-OFDM System Parameters
%<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
% Following parameters of the OFDM communication system are adopted
% from the conference version of the paper
fc = 13e3; %carrier frequency, Hz

B = 2.4425e3; %bandwidth, Hz

K = 256; %no. of subcarriers

Tg = 6.15e-3; %guard interval, seconds

% Derived parameters
T = K/B;                  % OFDM Symbol Duration

df = 1/T;                 % subcarrier spacing

f = fc + (-K/2:K/2-1)*df; % subcarrier frequencies
%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

%% 2. Transmitter/Source side parameters
%<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
%------------- (2a) Define M-QAM modulation order -------------------------
mQAM = 4; %No. of QAM symbols used in digital modulation

%------------- (2b) Define the Pilot & Null carrier arrangement------------
pilotSpacing = 4; %intra pilot spacing

L = pilotSpacing(1); %Pilot symbol is placed at every L'th subcarriers

% Pilot Carriers...
pilotData = qammod(mQAM/2, mQAM); %all pilots data are chosen to be the same symbol

pilotCarriers = 1 : L : K; %set of pilot subcarriers: uniformly spaced at every L'th subcarrier

% Null Carriers...
D = 0; % D null subcarriers are located on either side of a pilot subcarrier: pilotLocation + [ -D:-1 D:1]

nullData = 0; %all null carrier data are set to 0

Knulls = floor(96/1024*K);

Knulls_edges = floor(Knulls/3);

Knull_mid_start = floor(130/1024*K);

Knull_mid_end = floor(894/1024*K);

nullCarriers = setdiff([1:Knulls_edges... %start of the band edge
    Knull_mid_start:4*L:Knull_mid_end...  %middle of the band
    K-Knulls_edges+1:K], pilotCarriers);  %null carriers for ZERO Doppler

% Known Carriers = Pilots + Nulls...
knownCarriers = union(pilotCarriers, nullCarriers);

% Data Carriers...
dataCarriers = setdiff(1:K,knownCarriers); % Data Carriers: S_D = {1:K} \ (S_P U S_N)

% Load pilots and nulls & plot
data = zeros(K,1); % Vector of K symbols mounted on the OFDM subcarriers

data(pilotCarriers) = pilotData;  % load pilot data

data(nullCarriers) = nullData;    % load null data

% figure(1), stem(knownCarriers, abs( data(knownCarriers) )), grid on

measuredCarriers = pilotCarriers; %subcarrier indices at which measurements are made to estimate the channel

knownData = zeros(K,1); %buffer containing all unknown data set to zero

knownData(knownCarriers) = data(knownCarriers); %load all known data

%-----------(2c) Define and construct the Communication Objects------------
% Set the handles to the coder & decoder object to void
hTEnc = []; %handle to the coder is set empty

hTDec = []; %handle to the decoder is set empty

frmLen = log2(mQAM)*length(dataCarriers); %frame len in this case is simply the number of data bits

nUsed = length(dataCarriers); %all data carriers will be used in the uncoded case

dataCarriersUsed = dataCarriers( 1 : nUsed ); %subset of data carriers that are used for mounting encoded symbols
%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

%% 3. Propagation Channel Parameters
%<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
Npath = 4; % No. of propagation paths

mu = 1e-3; % mean of inter-arrival times (exp. distrib.), seconds
%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

%% 4. Receiver side parameters...
%<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
%--------------(4a) Partial Interval Demodulator parameters----------------
M = 4; %No. of Partial Intervals

%--------------(4b) Dictionary Parameters----------------------------------
lambda = 2; %delay grid oversampling factor: tau_p = T/(lambda*K):T/(lambda*K):Tg

delBmax = 10e-4/15; %spacing of Doppler scale grids (b_p)

[ tau_p, b_p ] = uwacDictionaryConfig( T, K, Tg, lambda, bmax(1), delBmax, fc, M, knownData, measuredCarriers, 3 );

Nt = length(tau_p); %Number of delay grids

Nb = length(b_p); %retrieve the number of Doppler grids

structChannel.Dictionary.lambda = lambda; %attach dictionary delay grid oversampling param to structChannel

structChannel.Dictionary.Nb = Nb; %attach dictionary Doppler grid number to structChannel

%---------------(4c) Receiver algorithm configuration----------------------
% Structure definitions for different channel estimation & data detection
% algorithms (Name, Model used for Channel Recovery, Model used for Data Detection)
structCSI.Name = 'CSI'; %structure: configurations of CSI Decoding
structCSI.ChannelRecovery.Model = []; %Perfect CSI : channel is known (a genie)
structCSI.DataDetection.Model = 'FID'; %CSI uses FID measurements for data detection

structLCEDD.Name = 'LS'; %structure: configurations of Least-Squares Channel Estimation & Decoding algorithm
structLCEDD.ChannelRecovery.Model = 'FID'; %LS estimates channel vector from FID output
structLCEDD.DataDetection.Model = 'FID'; %LS uses FID measurements for data detection

structBerger.Name = 'Berger'; %structure: configurations of Berger's algorithm
structBerger.SparseRecovery.Model = 'FID'; %Berger's algo estimates sparse channel vector from FID output
structBerger.DataDetection.Model = 'FID'; %Berger's algo uses FID measurements for data detection

structJCEDD_Stage_1.Name = 'Stage-1 of JCEDD'; %structure: configurations of the stage 1 of Joint Channel Estm & Data Det (JCEDD) algorithm
structJCEDD_Stage_1.SparseRecovery.Model = 'PID'; %In stage 1, JCEDD estimates sparse channel vector from PID output
structJCEDD_Stage_1.DataDetection.Model = 'PID'; %In stage 1, JCEDD algo uses FID measurements for data detection

structJCEDD_Stage_2.Name = 'Stage-2 of JCEDD'; %structure: configurations of the stage 2 of Joint Channel Estm & Data Det (JCEDD) algorithm
structJCEDD_Stage_2.SparseRecovery.Model = 'FID'; %In stage 2, JCEDD estimates sparse channel vector from FID output
structJCEDD_Stage_2.DataDetection.Model = 'PID'; %In stage 2, JCEDD algo uses PID measurements for data detection

% Structure definitions for sparse channel recovery algo configuration
structOMP.Name = 'OMP'; %sparse (channel) recovery algorithm used
structOMP.MaxPaths = Npath+3;
structOMP.Param = Npath+3;
structOMP.Iters = [];

structDummy.Name = ''; %LS & CSI does not use sparse recovery algo for channel estimation

%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

%% 5. Monte-Carlo run parameters & buffers...
%<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
% Setup the Monte Carlo (inner) loop variable to run across SNRs   
disp('Monte Carlo runs are scheduled across SNR');

loopVar = snr;

loopType = 'SNR';

xAxisTitle = ['SNR. b_{max} = ' num2str(bmax)];

xAxisLabel = 'SNR (dB)';

% Setup all the BER buffers
niters = 3; %number of iterations in the stage-1 and stage-2

%1 (Berger)
ber_CS_fullFftPilot = zeros(ntrials,length(loopVar)); %BER of CS algorithm using Pilot only O/P of Full FFT Demod

%3 (Stage-1 using OMP)
ber_OMP_Stage1_partFftPilot = zeros(ntrials,length(loopVar)); %BER of Stage-1 of the proposed algorithm (uses pilot only O/P of partial FFT Demod.)

%4 (Stage-2 using OMP)
ber_OMP_Stage2_fullFft = zeros(ntrials,length(loopVar)); %BER of Stage-2 of the proposed algorithm

%9 (LS)
ber_LS = zeros(ntrials,length(loopVar)); %BER of Least Squares data detection algo.

%10 (CSI)
ber_CSI = zeros(ntrials,length(loopVar)); %BER of genie aided data detector
%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

%% 6. Result logging file & rate
%<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
%Data logging file name: comprehensive file
resultFileName = ['uwacResult-M' num2str(M)...
    '-Niters' num2str(niters) '-Ntrials' num2str(ntrials) '-' algoList...
    '-Coding-OFF' '-' datestr(now,30) '.mat']; % all variables are saved to this file

%Meta data file: a text files that contains only a few bytes in a tabular
%format - useful for quick reading from a terminal
metafile = 'metadata.txt';
metaformat = '%2.4f';
metavars = loopType;
metadata = loopVar(:);

saveRate = 10; %all variables are saved after every saveRate (in minutes)

saveTime = 0; %Time (in minutes) since the last save
%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>%

%% 7. Monte-Carlo Runs...
%<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
nstart = 1; %start off with trial-1

uwacRun; %Invoke uwacRun.m to carry through Monte-Carlo Runs for various SNRs
% uwacRun.m saves the intermediate results during Monte-Carlo Runs, saves
% the final result and plots after completion of all trials
%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
