function [rPow, dPow, rPowm] = uwacStats(fc, K, T, M, Npath, Tg, mu, bmax, mQAM, knownCarriers, knownData, Ntrials)
% UWA Channel Statistics through Monte-Carlo Simulation
%   This function estimates the power of the signal component in the
%   received data y = Hs + w, where H is the simulated channel matrix
%   (realisation), s is the transmited data vector and w is the noise
%   vector. The power of the signal part is given by E{||Hs||^2} and its
%   estimated is found by averaging ||Hs||^2 across many realisations of H
%   and s drawn in a successive Monte-Carlo simulation run.
%
%INPUTS:
%   fc     : carrier frequency
%
%   K      : No. of Sub-carriers
%
%   T      : Duration of an OFDM symbol.
%
%   M      : Number of partial FFTS formed
%
%   Npath  : Number of propagation paths
%
%   Tg     : Guard Interval
%
%   mu     : Exponential PDF Parameter
%
%   bmax   : maximum doppler scale
%
%   SNR    : Signal to Noise Ratio = 20*log10(||Hs||/||n||)
%
%   mQAM   : Number of QAM symbols
%
%   knownData     : Kx1 data vector containing pilot and null carrier data
%                   alone and rest set to zero
%
%   knownCarriers : Location of known (pilot+null) carriers
%
%   Ntrials       : Number of Monte-Carlo runs to estimate received signal
%                   power
%
%
%OUTPUTS:
%   rPow   : Estimate of received signal power E{ ||Hs||^2
%
%
%REFERENCE:
% [1] Arunkumar K.P. and Chandra R. Murthy, "Iterative Sparse Channel
% Estimation and Data Detection for Underwater Acoustic Communications
% Using Partial Interval Demodulation",  IEEE Transactions on Signal
% Processing ( Volume: 66 , Issue: 19 , Oct.1, 1 2018 )
%
%
%Author  : Arunkumar K. P.
%Address : Ph.D. Scholar,
%          Signal Processing for Communications Lab, ECE Department,
%          Indian Institute of Science, Bangalore, India-560 012.
%Email   : arunkumar@iisc.ac.in
%
%REVISION HISTORY
% Version : 2.1
% Last Revision: 05-02-2018
%
%
% This script/program is released under the Commons Creative Licence
% with Attribution Non-commercial Share Alike (by-nc-sa)
% http://creativecommons.org/licenses/by-nc-sa/3.0/
%
% Short Disclaimer: this script is for educational purpose only.
%++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%% generate arrival times
interArrivalTimes = exprnd(mu,[1,Npath,Ntrials]);%inter arrival times ~ expon. with mean mu

interArrivalTimes = sort(interArrivalTimes,2);

tau_p0 = cumsum(interArrivalTimes,2);%arrival times at t=0

%% generate path attenuation coeffs
pathLoss_dB = tau_p0*(20/Tg);% Path attentuation (in dB)

pathLoss = 10.^(-pathLoss_dB/10);% Path attenuation (on linear scale)

A_p0 = sqrt(pathLoss) .* randn(1,Npath,Ntrials) ; %real path amplitudes: modified on 30/09/2017

%% generate doppler scales
b_p0 = bmax*2*(rand(1,Npath,Ntrials)-.5);

%% Generate data vector
Data = repmat(knownData(:), 1, Ntrials); % source data vector

dataCarriers = setdiff(1:K,knownCarriers);

Data(dataCarriers,:) = ...
    qammod( randi(mQAM,[length(dataCarriers),Ntrials]) - 1, mQAM ); %modulated random data

%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Data(knownCarriers,:) = 0; %discounting pilot power from SNR Calculation
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

%% simulate channel response
df = 1/T; %subcarrier spacing

f = fc + (-K/2:K/2-1)*df; %subcarriers

Zsm = zeros(K,Ntrials,M);

dT = T/M; %time-step from one partial FFT output to the next

kMat = bsxfun(@plus, -(0:K-1)', 0:K-1); %integer matrix: [kMat]l,k = k-l

for m = 1:M
    %% simulate received data from m'th partial FFT blocks
    tm = (m-1)*dT + dT/2; %time at the middle of partial FFT block
    
    tau_p_tm = tau_p0 - b_p0*tm; %Time-varying delay within an OFDM symbol at the m'th partial FFT block: tau_p(t) = tau_p - a_p*t
    
    phase_tm = bsxfun(@times, 2*pi*f(:), tau_p_tm);
    
    Gmat = exp(-1i*phase_tm);
    
    Jm = 1/M * exp( 1i*2*pi*kMat*(2*m-1)/(2*M) ) .* sinc( kMat/M );
    
    Qm = squeeze( sum(bsxfun(@times, Gmat, A_p0),2) );
    
    Zsm(:,:,m) = Jm * ( squeeze(Qm).*Data );
    
end

rPowm = squeeze( mean( sum( abs(Zsm.^2), 1 ), 2 ) );

Zs = sum( Zsm,3 );

rPow = mean( sum( abs(Zs).^2, 1 ), 2 );

dPow = mean( sum( abs(Data).^2, 1 ), 2 );

end