%%
% This script is to demonstrate the simulations in the paper: 
% J. Zhang, S. P. Chepuri, R. C. Hendriks, R. Heusdens, Microphone Subset
% Selection for MVDR Bemaformer Based Noise Reduction, IEEE/ACM Trans.
% Audio, Speech, Language Processing, 2018. (accepted)
% Contact: Jie Zhang (j.zhang-7@tudelft.nl), TU Delft, the Netherlands
%
% Requirements: RIR generator, e.g., https://www.audiolabs-erlangen.de/fau/professor/habets/software/rir-generator
% CVX toolbox: http://cvxr.com/cvx/download/, provided by S. Boyd at Stanford University
% Audio data, e.g., TIMIT database
% Here, we use the statistics in one frequency bin to show the performance,
% the output noise power or SNR should be averaged over all freqencies.

% Explanation for test_data.mat:
% M:     number of microphones
% L:     dimension of the environment
% FC:    location of the fusion center
% mpos:  positions of microphones
% sigpos:position of the target source
% npos1: position of the 1st interfering source
% npos2: position of the 2nd interfering source
% alpha: performance controllor, can be changed by user
% as:    acoustic transfer function
% c0:    squared distance from microphones to FC
% Rnn:   noise correlation matrix (M x M)
% Rxx:   correlation matrix of target signal (M x M)

%% Part I: model-driven sensor selection to generate the plots in Figure 2

close all, clear all, clc,
load('test_data.mat')
% alpha = 0.9; % this can be changed by user

b = min(abs(eig(Rnn)))/2;
S = Rnn - b * eye(M);
S = inv(S);
mvdr = inv(Rnn)*as/(as'*inv(Rnn)*as);
npower_mvdr = real(mvdr'*Rnn*mvdr);
lambda = alpha/npower_mvdr;

cvx_begin  SDP 
     variable p(M,1) 
%      variable Z(M,M) hermitian 
     minimize ( sum(diag(p)*c0) )
     subject to
%              trace(Z ) >= lambda
%              [S + 1/b*diag(p), S*srRxx; srRxx*S, srRxx*S*srRxx-Z] == hermitian_semidefinite(2*M)
             [S + 1/b*diag(p), S*as; as'*S, as'*S*as-lambda] == hermitian_semidefinite(1+M)
             p <= 1
             p >= 0
cvx_end
se_set = random_rounding(p, Rnn, as, 1/lambda)

figure  % to plot selected sensors and configuration
set(gca, 'fontsize', 14);
grid on; box on;
hold on;
xlabel('$x(m)$','FontSize',18,'Interpreter','Latex');
ylabel('$y(m)$','FontSize',18,'Interpreter','Latex');
axis([-0.3 L(1)+0.3 -0.3 L(2)+0.3])
plot(mpos(:,1),mpos(:,2), 'ok');
plot(sigpos(1), sigpos(2), 'or', 'MarkerSize',10,'MarkerFaceColor','r','markerEdgeColor','r');
plot([npos1(1) npos2(1)], [npos1(2) npos2(2)], '*b', 'MarkerSize',10,'MarkerFaceColor','b','markerEdgeColor','b')
plot(FC(1), FC(2), 'sk', 'MarkerSize',12,'MarkerFaceColor','k','markerEdgeColor','k')
plot(mpos(se_set,1),mpos(se_set,2), 'ob', 'MarkerSize',6,'MarkerFaceColor','b','markerEdgeColor','b')
legend('sensors','source','interference','FC','selected','FontSize',14,'Location','northoutside','Orientation','horizontal');
box on,                
% th = 0:pi/50:2*pi;
% xunit = 6 * cos(th) + FC(1); % radius = 2m
% yunit = 6 * sin(th) + FC(2);
% h = plot(xunit, yunit,'-g','LineWidth',2);

%% uncorrelated case: linear programming problem based on rank ordering
% sigma: noise variance
% v = c0.*sigma./abs(as).^2;
% [a,b] = sort(v,'ascend');
% indi = c0(b)./(a+eps);
% for i = 1:M
%     if sum(indi(1:i)) >= lambda
%         break;
%     end
% end